package api

import (
	"net/http"
	"time"

	"code.justin.tv/cb/roster/internal/authorization"
	"code.justin.tv/cb/roster/internal/clients/pushy"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/clients/users"
	"code.justin.tv/cb/roster/internal/httputil"
	"code.justin.tv/cb/roster/internal/liveline"
	goji "goji.io"
	"goji.io/pat"
)

const handlerTimeout = 5 * time.Second

// Server contains a router and client interfaces to downstream services.
type Server struct {
	mux         *goji.Mux
	authDecoder *authorization.Decoder
	cache       Cache
	dbWriter    DBWriter
	dbReader    DBReader
	liveline    liveline.Liveline
	s3          S3
	s3Uploader  S3Uploader
	users       users.Client
	pushy       pushy.Pushy
}

// ServerParams contains the dependencies for instantiating a Server.
type ServerParams struct {
	AuthDecoder      *authorization.Decoder
	Cache            Cache
	DBWriter         DBWriter
	DBReader         DBReader
	Liveline         liveline.Liveline
	S3               S3
	S3Uploader       S3Uploader
	Users            users.Client
	Pushy            pushy.Pushy
	TelemetryHandler telemetryhook.Handler
}

// NewServer instantiates a Server with the defined routes and corresponding handlers,
// and returns the Server.
func NewServer(params *ServerParams) *Server {
	server := &Server{
		mux:         goji.NewMux(),
		authDecoder: params.AuthDecoder,
		cache:       params.Cache,
		dbWriter:    params.DBWriter,
		dbReader:    params.DBReader,
		liveline:    params.Liveline,
		s3:          params.S3,
		s3Uploader:  params.S3Uploader,
		users:       params.Users,
		pushy:       params.Pushy,
	}

	server.mux.Use(httputil.PanicRecoveryMiddleware)
	server.mux.HandleFunc(pat.Get("/health"), httputil.HealthCheck)

	root := goji.SubMux()
	server.mux.Handle(pat.New("/*"), root)
	root.Use(httputil.AccessLogMiddleware)
	root.Use(params.TelemetryHandler.TelemetryMetricsHandler)
	root.Use(httputil.Timeout(handlerTimeout))

	v1 := goji.SubMux()
	root.Handle(pat.New("/v1/*"), v1)
	v1.HandleFunc(pat.Get("/search/teams"), server.searchV1Teams)
	v1.HandleFunc(pat.Get("/teams"), server.getV1Teams)

	team := goji.SubMux()
	v1.Handle(pat.New("/teams/:team_id"), team)
	team.Use(validateNumericTeamID)
	team.HandleFunc(pat.Get(""), server.getV1Team)
	team.HandleFunc(pat.Patch(""), server.patchV1Team)
	team.HandleFunc(pat.Delete(""), server.deleteV1Team)

	teams := goji.SubMux()
	v1.Handle(pat.New("/teams/:team_id/*"), teams)
	teams.Use(validateNumericTeamID)
	teams.HandleFunc(pat.Post("/image_uploads"), server.postV1ImageUploads)
	teams.HandleFunc(pat.Get("/invitations"), server.getV1TeamInvitations)
	teams.HandleFunc(pat.Post("/invitations"), server.postV1TeamInvitations)
	teams.HandleFunc(pat.Get("/memberships"), server.getV1TeamMemberships)
	teams.HandleFunc(pat.Post("/memberships"), server.postV1TeamMemberships)
	teams.HandleFunc(pat.Get("/live_memberships"), server.getV1TeamLiveMemberships)
	teams.HandleFunc(pat.Get("/featured_channels"), server.getV1FeaturedChannels)
	teams.HandleFunc(pat.Post("/featured_channels"), server.postV1FeaturedChannels)
	teams.HandleFunc(pat.Get("/live_featured_channels"), server.getV1LiveFeaturedChannels)

	teamsFeaturedChannels := goji.SubMux()
	teams.Handle(pat.New("/featured_channels/:channel_id"), teamsFeaturedChannels)
	teamsFeaturedChannels.Use(validateNumericChannelID)
	teamsFeaturedChannels.HandleFunc(pat.Patch(""), server.patchV1FeaturedChannel)
	teamsFeaturedChannels.HandleFunc(pat.Delete(""), server.deleteV1FeaturedChannel)

	teamChannels := goji.SubMux()
	teams.Handle(pat.New("/channels/:channel_id/*"), teamChannels)
	teamChannels.Use(validateNumericChannelID)
	teamChannels.HandleFunc(pat.Delete("/invitation"), server.deleteV1TeamInvitation)
	teamChannels.HandleFunc(pat.Patch("/membership"), server.patchV1TeamMembership)
	teamChannels.HandleFunc(pat.Delete("/membership"), server.deleteV1Membership)

	channels := goji.SubMux()
	v1.Handle(pat.New("/channels/:channel_id/*"), channels)
	channels.Use(validateNumericChannelID)
	channels.HandleFunc(pat.Get("/invitations"), server.getV1ChannelInvitations)
	channels.HandleFunc(pat.Get("/memberships"), server.getV1ChannelMemberships)

	channelTeams := goji.SubMux()
	channels.Handle(pat.New("/teams/:team_id/*"), channelTeams)
	channelTeams.Use(validateNumericTeamID)
	channelTeams.HandleFunc(pat.Put("/invitation"), server.putV1ChannelInvitation)
	channelTeams.HandleFunc(pat.Put("/membership"), server.putV1ChannelMembership)
	channelTeams.HandleFunc(pat.Delete("/membership"), server.deleteV1Membership)

	users := goji.SubMux()
	v1.Handle(pat.New("/users/:user_id/*"), users)
	users.Use(validateNumericUserID)
	users.HandleFunc(pat.Post("/teams"), server.postV1Teams)

	return server
}

// ServeHTTP allows Server to implement http.Handler.
func (s *Server) ServeHTTP(w http.ResponseWriter, req *http.Request) {
	s.mux.ServeHTTP(w, req)
}
