package api

import (
	"context"

	"code.justin.tv/cb/roster/internal/cache"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/s3"
	"code.justin.tv/cb/roster/internal/s3/uploader"
	"code.justin.tv/web/users-service/client/channels"
)

// Cache is the interface to a caching service.
type Cache interface {
	GetTeam(ctx context.Context, id string) (string, error)
	SetTeam(ctx context.Context, id, json string) error
	ClearTeam(ctx context.Context, id string) error

	GetTeams(ctx context.Context, query cache.TeamsQuery) (string, error)
	SetTeams(ctx context.Context, query cache.TeamsQuery, json string) error
	ClearAllTeams(ctx context.Context) error

	GetChannelMemberships(ctx context.Context, id string) (string, error)
	SetChannelMemberships(ctx context.Context, id, json string) error
	ClearChannelMemberships(ctx context.Context, id string) error

	GetTeamMemberships(ctx context.Context, query cache.TeamMembershipsQuery) (string, error)
	SetTeamMemberships(ctx context.Context, query cache.TeamMembershipsQuery, json string) error
	ClearAllTeamMembershipsForTeam(ctx context.Context, id string) error
}

// S3 is the app Server's expectation of the S3 service.
type S3 interface {
	Find(ctx context.Context, params s3.FindParams) (s3.Image, error)
	Save(ctx context.Context, params s3.SaveParams) error
}

// S3Uploader is the app Server's expectation of the S3 Uploader service.
type S3Uploader interface {
	NewImageUpload(params uploader.ImageUploadParams) (uploader.ImageUpload, error)
}

// DBWriter is how the Roster app expects the writing to DB to be.
type DBWriter interface {
	CreateInvitation(ctx context.Context, teamID, channelID string) error
	DeleteInvitation(ctx context.Context, teamID, channelID string) error
	AcceptInvitation(ctx context.Context, teamID, channelID string) error

	CreateMembership(ctx context.Context, membership db.Membership) error
	DeleteMembership(ctx context.Context, teamID, channelID string) error
	UpdateMembership(ctx context.Context, teamID, channelID string, revenueRevealed, statsRevealed bool) error
	UpdateMembershipDisplayPosition(ctx context.Context, teamID, channelID string, desiredPosition uint) error

	CreateTeam(ctx context.Context, params db.CreateTeamParams) (db.Team, error)
	UpdateTeam(ctx context.Context, team db.Team) error
	DeleteTeam(ctx context.Context, id string) error

	UpdateFeaturedChannelDisplayPosition(ctx context.Context, teamID, channelID string, desiredPosition uint) error
	CreateFeaturedChannel(ctx context.Context, teamID, channelID string) error
	DeleteFeaturedChannel(ctx context.Context, teamID, channelID string) error
}

// DBReader is how the Roster app expects reading from DB to be.
type DBReader interface {
	GetTeamInvitations(ctx context.Context, teamID string) ([]db.Invitation, error)
	GetInvitingTeamsForChannel(ctx context.Context, channelID string) ([]db.Team, error)
	GetInvitation(ctx context.Context, teamID, channelID string) (db.Invitation, error)

	GetTeamMemberships(ctx context.Context, teamID string, filter *db.MembershipsFilter) ([]db.Membership, error)
	GetChannelMemberships(ctx context.Context, channelID string) ([]db.Membership, error)
	GetMembership(ctx context.Context, teamID, channelID string) (db.Membership, error)

	GetTeamsCount(ctx context.Context, filter db.TeamsFilter) (uint, error)
	GetTeams(ctx context.Context, filter db.TeamsFilter) ([]db.Team, error)
	GetTeamByID(ctx context.Context, id string) (db.Team, error)
	SearchTeams(ctx context.Context, filter db.SearchTeamsFilter) ([]db.Team, error)
	SearchTeamsCount(ctx context.Context, filter db.SearchTeamsFilter) (uint, error)

	GetFeaturedChannels(ctx context.Context, teamID string) ([]db.FeaturedChannel, error)
	GetFeaturedChannel(ctx context.Context, teamID, channelID string) (db.FeaturedChannel, error)
}

// Users aliases the exported client interface from the Users service.
type Users interface {
	channels.Client
}
