package cache

import (
	"context"
	"fmt"

	"github.com/go-redis/redis"
	"github.com/pkg/errors"
)

// GetChannelMemberships attempts to retrieve the JSON channel memberships from the cache.
func (r *Redis) GetChannelMemberships(ctx context.Context, id string) (string, error) {
	json, err := r.client.WithContext(ctx).Get(channelMembershipsKey(id)).Result()
	if err != nil {
		switch err {
		case redis.Nil:
			return "", ErrNoChannelMemberships
		default:
			return "", errors.Wrap(err, "cache: failed to get channel memberships")
		}
	}

	return json, nil
}

// SetChannelMemberships sets (and potentially overwrites) JSON channel memberships in the cache.
func (r *Redis) SetChannelMemberships(ctx context.Context, id, json string) error {
	if len(json) == 0 {
		return nil
	}

	client := r.client.WithContext(ctx)
	err := client.Set(channelMembershipsKey(id), json, expiration).Err()
	if err != nil {
		return errors.Wrap(err, "cache: failed to set channel memberships")
	}

	return nil
}

// ClearChannelMemberships attempts to delete the JSON channel memberships from the cache.
func (r *Redis) ClearChannelMemberships(ctx context.Context, id string) error {
	err := r.client.WithContext(ctx).Del(channelMembershipsKey(id)).Err()
	if err != nil {
		return errors.Wrap(err, "cache: failed to clear channel memberships")
	}

	return nil
}

func channelMembershipsKey(id string) string {
	return fmt.Sprintf("channel:%s:memberships", id)
}
