package cache

import (
	"context"
	"fmt"

	"github.com/go-redis/redis"
	"github.com/pkg/errors"
)

// GetTeam attempts to retrieve the JSON team response from the cache.
func (r *Redis) GetTeam(ctx context.Context, id string) (string, error) {
	json, err := r.client.WithContext(ctx).Get(teamKey(id)).Result()
	if err != nil {
		switch err {
		case redis.Nil:
			return "", ErrNoTeam
		default:
			return "", errors.Wrap(err, "cache: failed to get team")
		}
	}

	return json, nil
}

// SetTeam sets (and potentially overwrites) a JSON team response in the cache.
func (r *Redis) SetTeam(ctx context.Context, id, json string) error {
	if len(json) == 0 {
		return nil
	}

	err := r.client.WithContext(ctx).Set(teamKey(id), json, expiration).Err()
	if err != nil {
		return errors.Wrap(err, "cache: failed to set team")
	}

	return nil
}

// ClearTeam attempts to delete the JSON team response from the cache.
func (r *Redis) ClearTeam(ctx context.Context, id string) error {
	err := r.client.WithContext(ctx).Del(teamKey(id)).Err()
	if err != nil {
		return errors.Wrap(err, "cache: failed to clear team")
	}

	return nil
}

func teamKey(id string) string {
	return fmt.Sprintf("team:%s", id)
}
