package cache

import (
	"context"
	"fmt"

	"github.com/go-redis/redis"
	"github.com/pkg/errors"
)

// TeamMembershipsQuery is used to construct the cache key.
type TeamMembershipsQuery struct {
	ID      string
	Revenue *bool
	Stats   *bool
}

// GetTeamMemberships attempts to retrieve the JSON team memberships from the cache.
func (r *Redis) GetTeamMemberships(ctx context.Context, query TeamMembershipsQuery) (string, error) {
	client := r.client.WithContext(ctx)
	json, err := client.Get(teamMembershipsKey(query)).Result()
	if err != nil {
		switch err {
		case redis.Nil:
			return "", ErrNoTeamMemberships
		default:
			return "", errors.Wrap(err, "cache: failed to get team memberships")
		}
	}

	return json, nil
}

// SetTeamMemberships sets (and potentially overwrites) JSON team memberships in the cache.
func (r *Redis) SetTeamMemberships(ctx context.Context, query TeamMembershipsQuery, json string) error {
	if len(json) == 0 {
		return nil
	}

	client := r.client.WithContext(ctx)
	indexKey := teamMembershipsIndexKey(query.ID)
	membershipsKey := teamMembershipsKey(query)

	err := client.SAdd(indexKey, membershipsKey).Err()
	if err != nil {
		return errors.Wrap(err, "cache: failed to sadd team memberships key to index")
	}

	err = client.Expire(indexKey, expiration).Err()
	if err != nil {
		return errors.Wrap(err, "cache: failed to expire team memberships index")
	}

	err = client.Set(membershipsKey, json, expiration).Err()
	if err != nil {
		return errors.Wrap(err, "cache: failed to set team memberships")
	}

	return nil
}

// ClearAllTeamMembershipsForTeam attempts to delete ALL cached JSON team memberships.
func (r *Redis) ClearAllTeamMembershipsForTeam(ctx context.Context, id string) error {
	client := r.client.WithContext(ctx)
	indexKey := teamMembershipsIndexKey(id)

	keys, err := client.SMembers(indexKey).Result()
	if err != nil {
		return errors.Wrap(err, "cache: failed to smembers all keys for team memberships")
	}

	allKeys := append([]string{indexKey}, keys...)

	err = client.Del(allKeys...).Err()
	if err != nil {
		return errors.Wrap(err, "cache: failed to delete team memberships")
	}

	return nil
}

func teamMembershipsIndexKey(id string) string {
	return fmt.Sprintf("team:%s:memberships:index", id)
}

func teamMembershipsKey(query TeamMembershipsQuery) string {
	return fmt.Sprintf(
		"team:%s:memberships:revenue:%s:stats:%s",
		query.ID,
		formatBoolPointer(query.Revenue),
		formatBoolPointer(query.Stats),
	)
}

func formatBoolPointer(pointer *bool) string {
	if pointer == nil {
		return "nil"
	}

	return fmt.Sprint(*pointer)
}
