package cache

import (
	"context"
	"fmt"

	"github.com/go-redis/redis"
	"github.com/pkg/errors"
)

const teamsIndexKey = "teams:index"

// TeamsQuery is used to construct the cache key.
type TeamsQuery struct {
	Limit            *uint
	Name             string
	Offset           uint
	OrderByDirection string
	UserID           string
}

// GetTeams attempts to retrieve the JSON teams from the cache.
func (r *Redis) GetTeams(ctx context.Context, query TeamsQuery) (string, error) {
	client := r.client.WithContext(ctx)

	json, err := client.Get(teamsKey(query)).Result()
	if err != nil {
		switch err {
		case redis.Nil:
			return "", ErrNoTeams
		default:
			return "", errors.Wrap(err, "cache: failed to get teams")
		}
	}

	return json, nil
}

// SetTeams sets (and potentially overwrites) JSON teams in the cache.
func (r *Redis) SetTeams(ctx context.Context, query TeamsQuery, json string) error {
	if len(json) == 0 {
		return nil
	}

	client := r.client.WithContext(ctx)
	key := teamsKey(query)

	err := client.SAdd(teamsIndexKey, key).Err()
	if err != nil {
		return errors.Wrap(err, "cache: failed to sadd teams key to index")
	}

	err = client.Expire(teamsIndexKey, expiration).Err()
	if err != nil {
		return errors.Wrap(err, "cache: failed to expire teams index")
	}

	err = client.Set(key, json, expiration).Err()
	if err != nil {
		return errors.Wrap(err, "cache: failed to set teams")
	}

	return nil
}

// ClearAllTeams attempts to delete ALL cached JSON teams.
func (r *Redis) ClearAllTeams(ctx context.Context) error {
	client := r.client.WithContext(ctx)

	keys, err := client.SMembers(teamsIndexKey).Result()
	if err != nil {
		return errors.Wrap(err, "cache: failed to smembers all keys for teams")
	}

	allKeys := append([]string{teamsIndexKey}, keys...)

	err = client.Del(allKeys...).Err()
	if err != nil {
		return errors.Wrap(err, "cache: failed to delete teams")
	}

	return nil
}

func teamsKey(query TeamsQuery) string {
	limit := "nil"
	if query.Limit != nil {
		limit = fmt.Sprint(*query.Limit)
	}

	return fmt.Sprintf(
		"teams:limit:%s:name:%s:offset:%d:order-by-direction:%s:user-id:%s",
		limit,
		query.Name,
		query.Offset,
		query.OrderByDirection,
		query.UserID,
	)
}
