package pdms

import (
	"context"
	"net/http"
	"time"

	"code.justin.tv/amzn/PDMSLambdaTwirp"
	twirplambdatransport "code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/endpoints"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/lambda"
	"github.com/golang/protobuf/ptypes"
	"github.com/pkg/errors"
)

// Service ID from Catalog https://catalog.xarth.tv/services/297/details
// This is the ID that PDMS uses when allowlisting services.
const serviceID string = "297"

// Client struct
type Client struct {
	client PDMSLambdaTwirp.PDMSService
}

// ClientConfig to instantiate a client
type ClientConfig struct {
	CallerRoleArn string
	LambdaArn     string
}

// NewClient instantiates a new client to communicate with PDMS
func NewClient(config ClientConfig) *Client {
	sess := session.Must(session.NewSessionWithOptions(session.Options{
		Config: aws.Config{
			Region:              aws.String("us-west-2"),
			HTTPClient:          &http.Client{Timeout: 10 * time.Second},
			STSRegionalEndpoint: endpoints.RegionalSTSEndpoint,
		},
	}))

	credentials := stscreds.NewCredentials(sess, config.CallerRoleArn)
	pdmsLambdaCli := lambda.New(sess, &aws.Config{Credentials: credentials})
	pdmsTransport := twirplambdatransport.NewClient(pdmsLambdaCli, config.LambdaArn)
	client := PDMSLambdaTwirp.NewPDMSServiceProtobufClient("", pdmsTransport)

	return &Client{client: client}
}

// PromiseDeletion reports a deletion to PDMS for Roster
func (c *Client) PromiseDeletion(context context.Context, userID string) error {
	timestamp, err := ptypes.TimestampProto(time.Now().Add(time.Hour * 24 * 7))
	if err != nil {
		return errors.Wrap(err, "error creating timestamp")
	}

	request := PDMSLambdaTwirp.PromiseDeletionRequest{
		UserId:      userID,
		ServiceId:   serviceID,
		Timestamp:   timestamp,
		AutoResolve: true,
	}

	if _, err := c.client.PromiseDeletion(context, &request); err != nil {
		return errors.Wrap(err, "pdms: failed to call promise deletion")
	}

	return nil
}
