package pushy

import (
	"context"
	"net/http"

	receiver "code.justin.tv/amzn/TwitchDartReceiverTwirp"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/cb/roster/config"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	log "github.com/sirupsen/logrus"
)

type dartClient struct {
	receiver receiver.Receiver
	reporter telemetry.SampleReporter
}

// NewDartClient creates a new pushy publisher client using DART.
func NewDartClient(httpClient *http.Client, reporter telemetry.SampleReporter) Pushy {
	client := receiver.NewReceiverProtobufClient(config.Values.ServiceHosts.Dart, httpClient)
	return &dartClient{
		receiver: client,
		reporter: reporter,
	}
}

func (c *dartClient) publishEvent(ctx context.Context, teamName, invitedUserID, eventName, senderID, recipientID string) error {
	request := &receiver.PublishNotificationRequest{
		NotificationType: eventName,
		SenderId:         senderID,
		Recipient: &receiver.PublishNotificationRequest_RecipientId{
			RecipientId: recipientID,
		},
		Metadata: map[string]string{
			"team_name":       teamName,
			"invited_user_id": invitedUserID,
		},
	}

	statRes := "success"

	resp, err := c.receiver.PublishNotification(ctx, request)
	if err != nil {
		statRes = "error"
	} else if resp != nil {
		log.WithFields(log.Fields{
			"notification_type":     eventName,
			"notification_trace_id": resp.NotificationTraceId,
		}).Info("dart: published notification")
	}

	reporter := telemetryhook.ReporterWithDimensions(map[string]string{"DartEvent": eventName}, c.reporter)
	reporter.Report(statRes, 1.0, telemetry.UnitCount)
	return err
}

// PublishInvite sends a notification to user <userID> via DART indicating they have
// been invited to team <teamName>.
func (c *dartClient) PublishInvite(ctx context.Context, teamUserID, teamName, userID string) error {
	return c.publishEvent(ctx, teamName, userID, "team_invite", teamUserID, userID)
}

// PublishInviteDeclined sends a notification via DART to the team owner <teamUserID>,
// indicating user <userID> has declined their invitation.
func (c *dartClient) PublishInviteDeclined(ctx context.Context, teamUserID, teamName, userID string) error {
	return c.publishEvent(ctx, teamName, userID, "team_invite_decline", userID, teamUserID)
}

// PublishInviteDeclined sends a notification via DART to the team owner <teamUserID>,
// indicating user <userID> has accepted their invitation.
func (c *dartClient) PublishInviteAccepted(ctx context.Context, teamUserID, teamName, userID string) error {
	return c.publishEvent(ctx, teamName, userID, "team_invite_accept", userID, teamUserID)
}
