package telemetryhook

import (
	"net/http"
	"strconv"
	"strings"
	"time"
)

type responseWriter struct {
	http.ResponseWriter
	status int
}

// TelemetryMetricsHandler is an HTTP middleware that logs the request.
func (c *Client) TelemetryMetricsHandler(inner http.Handler) http.Handler {
	middleware := func(w http.ResponseWriter, r *http.Request) {
		start := time.Now()
		logged := &responseWriter{
			ResponseWriter: w,
		}

		inner.ServeHTTP(logged, r)

		duration := time.Since(start)
		statName := buildStatName(r)

		reporter := ReporterWithDimensions(map[string]string{"Operation": statName, "StatusCode": strconv.Itoa(logged.status)}, c.Reporter)
		reporter.ReportDurationSample("Duration", duration)
	}

	return http.HandlerFunc(middleware)
}

func buildStatName(r *http.Request) string {
	var statNameArr []string
	splitStr := strings.Split(strings.Trim(r.URL.Path, "/"), "/")

	// Strip any numbers that might be id's out of the stat name
	for _, str := range splitStr {
		if _, err := strconv.Atoi(str); err != nil {
			statNameArr = append(statNameArr, str)
		}
	}

	statName := strings.Join(statNameArr, "_")
	statName = r.Method + ("_" + statName)

	return statName
}

func (rw *responseWriter) WriteHeader(status int) {
	rw.status = status
	rw.ResponseWriter.WriteHeader(status)
}

// TelemetryMetricsHandler is an HTTP middleware that does nothing.
func (c *NoopClient) TelemetryMetricsHandler(inner http.Handler) http.Handler {
	return inner
}
