package users

import (
	"context"

	"github.com/cep21/circuit"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/hygienic/errors"
	"code.justin.tv/web/users-service/client/channels"
	"code.justin.tv/web/users-service/models"
)

// Client represents a users service client.
type Client interface {
	Get(ctx context.Context, id string, reqOpts *twitchclient.ReqOpts) (*channels.Channel, error)
	GetByIDAndParams(ctx context.Context, id string, params *models.ChannelFilterParams, reqOpts *twitchclient.ReqOpts) (*channels.Channel, error)
	GetAll(ctx context.Context, ids []string, reqOpts *twitchclient.ReqOpts) (*channels.ChannelsResult, error)
	Set(ctx context.Context, up models.UpdateChannelProperties, reqOpts *twitchclient.ReqOpts) error
}

// UserCircuits holds the circuits used for relevant method calls.
type UserCircuits struct {
	Get              *circuit.Circuit
	GetAll           *circuit.Circuit
	GetByIDAndParams *circuit.Circuit
	Set              *circuit.Circuit
}
type clientImpl struct {
	usersClient channels.Client
	circuit     UserCircuits
}

func (c *clientImpl) Get(ctx context.Context, id string, reqOpts *twitchclient.ReqOpts) (*channels.Channel, error) {
	// NOTE: we intentionally do not treat "userID invalid" as a bad request.
	// this will cause the circuit to break if we start passing invalid IDs
	var userResult *channels.Channel
	if err := c.circuit.Get.Run(ctx, func(ctx context.Context) error {
		var innerErr error
		userResult, innerErr = c.usersClient.Get(ctx, id, reqOpts)
		return wrapUsersError(innerErr)
	}); err != nil {
		return nil, errors.Wrapf(err, "failure in users service")
	}

	return userResult, nil
}

func (c *clientImpl) GetByIDAndParams(ctx context.Context, id string, params *models.ChannelFilterParams, reqOpts *twitchclient.ReqOpts) (*channels.Channel, error) {
	// NOTE: we intentionally do not treat "userID invalid" as a bad request.
	// this will cause the circuit to break if we start passing invalid IDs
	var userResult *channels.Channel
	if err := c.circuit.GetByIDAndParams.Run(ctx, func(ctx context.Context) error {
		var innerErr error
		userResult, innerErr = c.usersClient.GetByIDAndParams(ctx, id, params, reqOpts)
		return wrapUsersError(innerErr)
	}); err != nil {
		return nil, errors.Wrapf(err, "failure in users service")
	}

	return userResult, nil
}

func (c *clientImpl) GetAll(ctx context.Context, ids []string, reqOpts *twitchclient.ReqOpts) (*channels.ChannelsResult, error) {
	// NOTE: we intentionally do not treat "userID invalid" as a bad request.
	// this will cause the circuit to break if we start passing invalid IDs
	var userResult *channels.ChannelsResult
	if err := c.circuit.GetAll.Run(ctx, func(ctx context.Context) error {
		var innerErr error
		userResult, innerErr = c.usersClient.GetAll(ctx, ids, reqOpts)
		return wrapUsersError(innerErr)
	}); err != nil {
		return nil, errors.Wrapf(err, "failure in users service")
	}

	return userResult, nil
}

func (c *clientImpl) Set(ctx context.Context, up models.UpdateChannelProperties, reqOpts *twitchclient.ReqOpts) error {
	// NOTE: we intentionally do not treat "userID invalid" as a bad request.
	// this will cause the circuit to break if we start passing invalid IDs
	if err := c.circuit.Set.Run(ctx, func(ctx context.Context) error {
		return wrapUsersError(c.usersClient.Set(ctx, up, reqOpts))
	}); err != nil {
		return errors.Wrapf(err, "failure in users service")
	}

	return nil
}

// NewClient creates a new users service client.
func NewClient(host string, circuit UserCircuits) (Client, error) {
	usersClient, err := channels.NewClient(twitchclient.ClientConf{
		Host: host,
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 100,
		},
	})

	if err != nil {
		return nil, errors.Wrapf(err, "error creating users client")
	}

	return &clientImpl{
		usersClient: usersClient,
		circuit:     circuit,
	}, nil
}
