package db

import (
	"context"
	"database/sql"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// AcceptInvitation makes an INSERT query to DB to create a record
// in `team_users` for a given team ID and user ID. It then makes a DELETE
// query to DB to delete a record in `team_invitations` for a given team ID and user ID.
func (c *Client) AcceptInvitation(ctx context.Context, teamID, channelID string) error {
	tx, err := c.db.BeginTx(ctx, nil)
	if err != nil {
		return errors.Wrap(err, "db: failed to begin transaction for invitation deletion and membership creation")
	}

	defer func() {
		if err == nil {
			return
		}

		if err = tx.Rollback(); err != nil {
			log.WithError(err).WithFields(log.Fields{
				"team_id":    teamID,
				"channel_id": channelID,
			}).Error("db: failed to rollback transaction for invitation deletion and membership creation")
		}
	}()

	if err = txInsertMembership(ctx, tx, teamID, channelID); err != nil {
		return err
	}

	if err = txDeleteInvitation(ctx, tx, teamID, channelID); err != nil {
		return err
	}

	if err = tx.Commit(); err != nil {
		return errors.Wrap(err, "db: failed to commit transaction")
	}

	return err
}

func txInsertMembership(ctx context.Context, tx *sql.Tx, teamID, channelID string) error {
	insert := `
		INSERT INTO team_users (team_id, user_id, display_order, created_at)
		SELECT $1, $2, COALESCE(MAX(display_order) + 1, 0), CURRENT_TIMESTAMP
		FROM team_users
		WHERE team_id = $1
		AND NOT EXISTS (
			SELECT *
			FROM team_users
			WHERE team_id = $1
			AND user_id = $2
		)
	`

	result, err := tx.ExecContext(ctx, insert, teamID, channelID)
	if err != nil {
		return errors.Wrap(err, "db: failed to create membership")
	}

	rowsAffected, err := result.RowsAffected()
	if err != nil || rowsAffected < 1 {
		return ErrNoMembershipCreated
	}

	return nil
}

func txDeleteInvitation(ctx context.Context, tx *sql.Tx, teamID, channelID string) error {
	delete := "DELETE FROM team_invitations WHERE team_id = $1 AND user_id = $2"

	result, err := tx.ExecContext(ctx, delete, teamID, channelID)
	if err != nil {
		return errors.Wrap(err, "db: failed to delete row from team_invitations")
	}

	rowsAffected, err := result.RowsAffected()
	if err != nil || rowsAffected < 1 {
		return ErrNoRowFoundForDeletion
	}

	return nil
}
