package db

import (
	"context"
	"errors"

	"database/sql"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("putChannelInvitation", func() {
	var (
		db                         *Client
		mock                       sqlmock.Sqlmock
		teamID                     string
		channelID                  string
		createMembershipQueryRegEx string
		deleteInvitationQueryRegEx string
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		postgresDB := &postgres.DB{
			DB: stub,
		}

		db = &Client{
			db: postgresDB,
			displayPositioner: &displayPositionHelper{
				db: postgresDB,
			},
		}

		teamID = "123"
		channelID = "999999999"
		createMembershipQueryRegEx = `
	  		INSERT INTO team_users \(team_id, user_id, display_order, created_at\)
	  		SELECT \$1, \$2, COALESCE\(MAX\(display_order\) \+ 1, 0\), CURRENT_TIMESTAMP
	  		FROM team_users
	  		WHERE team_id = \$1
				AND NOT EXISTS \( SELECT \* FROM team_users WHERE team_id = \$1 AND user_id = \$2 \)
  		`

		deleteInvitationQueryRegEx = `DELETE FROM team_invitations WHERE team_id = \$1 AND user_id = \$2`
	})

	It("errors when beginning the transaction fails", func() {
		mock.ExpectBegin().WillReturnError(errors.New("transaction has not begun"))

		err := db.AcceptInvitation(context.Background(), teamID, channelID)
		Expect(err.Error()).To(ContainSubstring("db: failed to begin transaction for invitation deletion and membership creation: transaction has not begun"))
		mockError := mock.ExpectationsWereMet()
		Expect(mockError).NotTo(HaveOccurred())
	})

	Context("when create member query fails", func() {
		BeforeEach(func() {
			mock.ExpectBegin()
			mock.ExpectExec(createMembershipQueryRegEx).
				WithArgs(teamID, channelID).
				WillReturnError(errors.New("create membership has failed"))
			mock.ExpectRollback()
		})

		It("errors when create member query fails", func() {
			err := db.AcceptInvitation(context.Background(), teamID, channelID)

			Expect(err).To(HaveOccurred())
			Expect(err.Error()).To(ContainSubstring("db: failed to create membership: create membership has failed"))
			mockError := mock.ExpectationsWereMet()
			Expect(mockError).NotTo(HaveOccurred())
		})
	})

	Context("when create membership query succeeds", func() {
		BeforeEach(func() {
			mock.ExpectBegin()
			mock.ExpectExec(createMembershipQueryRegEx).
				WithArgs(teamID, channelID).
				WillReturnResult(sqlmock.NewResult(0, 1))
		})

		Context("when delete invitation query fails", func() {
			BeforeEach(func() {
				mock.ExpectExec(deleteInvitationQueryRegEx).WithArgs().WillReturnError(errors.New("some error"))
				mock.ExpectRollback()
			})

			It("errors", func() {
				err := db.AcceptInvitation(context.Background(), teamID, channelID)

				Expect(err).To(HaveOccurred())
				Expect(err.Error()).To(ContainSubstring("db: failed to delete row from team_invitations"))
				mockError := mock.ExpectationsWereMet()
				Expect(mockError).NotTo(HaveOccurred())
			})
		})

		Context("when delete invitation query succeeds", func() {
			BeforeEach(func() {
				mock.ExpectExec(deleteInvitationQueryRegEx).
					WithArgs(teamID, channelID).
					WillReturnResult(sqlmock.NewResult(0, 1))
			})

			Context("When commit fails", func() {
				BeforeEach(func() {
					mock.ExpectCommit().WillReturnError(errors.New("some commit error"))
				})

				It("errors", func() {
					err := db.AcceptInvitation(context.Background(), teamID, channelID)

					Expect(err).To(HaveOccurred())
					Expect(err.Error()).To(ContainSubstring("db: failed to commit transaction: some commit error"))
					mockError := mock.ExpectationsWereMet()
					Expect(mockError).NotTo(HaveOccurred())
				})
			})

			Context("when commit succeeds", func() {
				BeforeEach(func() {
					mock.ExpectCommit()
				})

				It("return nil", func() {
					err := db.AcceptInvitation(context.Background(), teamID, channelID)

					Expect(err).NotTo(HaveOccurred())
					mockError := mock.ExpectationsWereMet()
					Expect(mockError).NotTo(HaveOccurred())
				})
			})
		})
	})
})
