package db

import (
	"code.justin.tv/cb/roster/internal/postgres"
	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

const defaultMaxConns = 50
const defaultMaxIdleConns = 10

// Client is a wrapper for sql.DB.
type Client struct {
	db                *postgres.DB
	displayPositioner DisplayPositioner
}

// ClientConfig are the configuration values needed to connect to PostgreSQL.
type ClientConfig struct {
	Host         string
	Port         string
	DBName       string
	User         string
	Password     string
	SSLMode      string
	MaxConns     int
	MaxIdleConns int
}

// NewClient opens a connection pool to the PostgreSQL database,
// specified by the database credentials and address.
//
// Pinging the database ensures that the connection is established.
func NewClient(config ClientConfig) (*Client, error) {
	db, err := postgres.NewDB(postgres.DBConfig{
		Host:     config.Host,
		Port:     config.Port,
		DBName:   config.DBName,
		User:     config.User,
		Password: config.Password,
		SSLMode:  config.SSLMode,
	})

	if err != nil {
		return nil, errors.Wrap(err, "db: failed to instantiate client")
	}

	conns := defaultMaxConns
	if config.MaxConns > 0 {
		conns = config.MaxConns
	}

	idleConns := defaultMaxIdleConns
	if config.MaxIdleConns > 0 {
		idleConns = config.MaxIdleConns
	}
	db.SetMaxOpenConns(conns)
	db.SetMaxIdleConns(idleConns)

	log.Infof(
		"db: opened connection to postgresql database at %s:%s/%s",
		config.Host,
		config.Port,
		config.DBName,
	)

	return &Client{
		db: db,
		displayPositioner: &displayPositionHelper{
			db: db,
		},
	}, nil
}
