package db

import (
	"context"
	"database/sql"
	"errors"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("CreateFeaturedChannel", func() {
	var (
		db               *Client
		mock             sqlmock.Sqlmock
		teamID           string
		channelID        string
		insertQueryRegEx string
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		db = &Client{
			db: &postgres.DB{
				DB: stub,
			},
		}

		teamID = "123"
		channelID = "999999999"
		insertQueryRegEx = `
			INSERT INTO team_streams \(team_id, user_id, display_order, created_at\)
			SELECT \$1, \$2, COALESCE\(MAX\(display_order\) \+ 1, 0\), CURRENT_TIMESTAMP
			FROM team_streams
			WHERE team_id = \$1
			AND NOT EXISTS \( SELECT \* FROM team_streams WHERE team_id = \$1 AND user_id = \$2 \)
		`
	})

	It("errors when creating featured channel fails", func() {
		mock.ExpectExec(insertQueryRegEx).
			WithArgs(teamID, channelID).
			WillReturnError(errors.New("some error"))

		err := db.CreateFeaturedChannel(context.Background(), teamID, channelID)
		Expect(err).To(HaveOccurred())
		Expect(err.Error()).To(Equal("db: failed to insert row into team_streams: some error"))

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	It("errors when creating featured channel returns no rows", func() {
		mock.ExpectExec(insertQueryRegEx).
			WithArgs(teamID, channelID).
			WillReturnResult(sqlmock.NewResult(0, 0))

		err := db.CreateFeaturedChannel(context.Background(), teamID, channelID)
		Expect(err).To(Equal(ErrNoFeaturedChannelCreated))

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	Context("when creating featured channel succeeds", func() {
		BeforeEach(func() {
			mock.ExpectExec(insertQueryRegEx).
				WithArgs(teamID, channelID).
				WillReturnResult(sqlmock.NewResult(0, 1))
		})

		It("returns no error", func() {
			err := db.CreateFeaturedChannel(context.Background(), teamID, channelID)
			Expect(err).NotTo(HaveOccurred())

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})
	})
})
