package db

import (
	"context"
	"database/sql"
	"errors"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("CreateInvitation", func() {
	var (
		db         *Client
		mock       sqlmock.Sqlmock
		teamID     string
		channelID  string
		queryRegEx string
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		db = &Client{
			db: &postgres.DB{
				DB: stub,
			},
		}

		teamID = "123"
		channelID = "999999999"

		queryRegEx = `
			INSERT INTO team_invitations \(team_id, user_id, created_on\)
			SELECT \$1, \$2, CURRENT_TIMESTAMP
			WHERE NOT EXISTS \(
				SELECT \*
				FROM team_invitations
				WHERE team_id = \$1
				AND user_id = \$2
			\)`
	})

	It("errors when creating invitation fails", func() {
		mock.ExpectExec(queryRegEx).
			WithArgs(teamID, channelID).
			WillReturnError(errors.New("some error"))

		err := db.CreateInvitation(context.Background(), teamID, channelID)
		Expect(err).To(HaveOccurred())

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	It("errors when creating invitation returns no rows", func() {
		mock.ExpectExec(queryRegEx).
			WithArgs(teamID, channelID).
			WillReturnResult(sqlmock.NewResult(0, 0))

		err := db.CreateInvitation(context.Background(), teamID, channelID)
		Expect(err).To(Equal(ErrNoInvitationCreated))

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	Context("when creating invitation succeeds", func() {
		BeforeEach(func() {
			mock.ExpectExec(queryRegEx).
				WithArgs(teamID, channelID).
				WillReturnResult(sqlmock.NewResult(0, 1))
		})

		It("returns no error", func() {
			err := db.CreateInvitation(context.Background(), teamID, channelID)
			Expect(err).NotTo(HaveOccurred())

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})
	})
})
