package db

import (
	"context"

	"github.com/pkg/errors"
)

// CreateMembership makes a INSERT query to DB to create a record
// in `team_users` for a given team ID and user id.
func (c *Client) CreateMembership(ctx context.Context, membership Membership) error {
	statement := `
		INSERT INTO team_users (team_id, user_id, display_order, view_revenue, view_stats, created_at)
		SELECT $1, $2, COALESCE(MAX(display_order) + 1, 0), $3, $4, CURRENT_TIMESTAMP
		FROM team_users
		WHERE team_id = $1
		AND NOT EXISTS (
			SELECT *
			FROM team_users
			WHERE team_id = $1
			AND user_id = $2
		)
	`

	result, err := c.db.ExecContext(
		ctx,
		statement,
		membership.TeamID,
		membership.ChannelID,
		membership.RevenueRevealed,
		membership.StatsRevealed,
	)
	if err != nil {
		return errors.Wrap(err, "db: failed to insert row into team_users")
	}

	rowsAffected, err := result.RowsAffected()
	if err != nil || rowsAffected < 1 {
		return ErrNoMembershipCreated
	}

	return nil
}
