package db

import (
	"context"
	"database/sql"
	"errors"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("CreateMembership", func() {
	var (
		db                             *Client
		mock                           sqlmock.Sqlmock
		teamID, channelID              string
		revenueRevealed, statsRevealed bool
		membership                     Membership
		queryRegEx                     string
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		db = &Client{
			db: &postgres.DB{
				DB: stub,
			},
		}

		teamID = "123"
		channelID = "999999999"
		revenueRevealed = true
		statsRevealed = true

		membership = Membership{
			TeamID:          teamID,
			ChannelID:       channelID,
			RevenueRevealed: revenueRevealed,
			StatsRevealed:   statsRevealed,
		}

		queryRegEx = `
			INSERT INTO team_users \(team_id, user_id, display_order, view_revenue, view_stats, created_at\)
			SELECT \$1, \$2, COALESCE\(MAX\(display_order\) \+ 1, 0\), \$3, \$4, CURRENT_TIMESTAMP
			FROM team_users
			WHERE team_id = \$1
			AND NOT EXISTS \( SELECT \* FROM team_users WHERE team_id = \$1 AND user_id = \$2 \)
		`
	})

	It("errors when creating membership fails", func() {
		mock.ExpectExec(queryRegEx).
			WithArgs(teamID, channelID, revenueRevealed, statsRevealed).
			WillReturnError(errors.New("some error"))

		err := db.CreateMembership(context.Background(), membership)
		Expect(err).To(HaveOccurred())

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	It("errors when creating membership returns no rows", func() {
		mock.ExpectExec(queryRegEx).
			WithArgs(teamID, channelID, revenueRevealed, statsRevealed).
			WillReturnResult(sqlmock.NewResult(0, 0))

		err := db.CreateMembership(context.Background(), membership)
		Expect(err).To(Equal(ErrNoMembershipCreated))

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	Context("when creating membership succeeds", func() {
		BeforeEach(func() {
			mock.ExpectExec(queryRegEx).
				WithArgs(teamID, channelID, revenueRevealed, statsRevealed).
				WillReturnResult(sqlmock.NewResult(0, 1))
		})

		It("returns no error", func() {
			err := db.CreateMembership(context.Background(), membership)
			Expect(err).NotTo(HaveOccurred())

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})
	})
})
