package db

import (
	"context"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// CreateTeamParams contains fields for creating a Team in DB.
type CreateTeamParams struct {
	Name                string
	DisplayName         string
	UserID              string
	DescriptionMarkdown *string // optional
}

// CreateTeam makes a INSERT query to DB to create a record in `teams`.
func (c *Client) CreateTeam(ctx context.Context, params CreateTeamParams) (Team, error) {
	// The `WHERE NOT EXISTS` statement is required
	// because there is no unique index on `teams.name`
	statement := `
		INSERT INTO teams (name, display_name, user_id, info, created_at)
		SELECT $1::varchar, $2, $3, $4, CURRENT_TIMESTAMP
		WHERE NOT EXISTS (SELECT * FROM teams WHERE name = $1)
		RETURNING
			id, name, display_name, user_id, info, team_logo_image, banner_image, background_image, created_at, updated_at
	`

	var info *string
	if params.DescriptionMarkdown != nil {
		var placeholder Team
		placeholder.SetDescriptionWithMarkdown(*params.DescriptionMarkdown)
		info = &placeholder.descriptionHTML
	}

	rows, err := c.db.QueryContext(ctx, statement, params.Name, params.DisplayName, params.UserID, info)
	if err != nil {
		return Team{}, errors.Wrap(err, "db: failed to insert row into teams")
	}

	if !rows.Next() {
		return Team{}, ErrNoTeamCreated
	}

	defer func() {
		if err = rows.Close(); err != nil {
			log.WithError(err).Error("db: failed to close rows for teams")
		}
	}()

	return teamFromRows(rows)
}
