package db

import (
	"context"
	"database/sql"
	"errors"
	"time"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("CreateTeam", func() {
	var (
		db         *Client
		mock       sqlmock.Sqlmock
		params     CreateTeamParams
		queryRegEx string

		teamID, name, displayName, userID, descriptionMarkdown, descriptionHTML string
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		db = &Client{
			db: &postgres.DB{
				DB: stub,
			},
		}

		teamID = "123"
		name = "the_avengers"
		displayName = "The Avengers"
		userID = "999999999"
		descriptionMarkdown = "# Hi"
		descriptionHTML = "<h1>Hi</h1>\n"

		params = CreateTeamParams{
			Name:                name,
			DisplayName:         displayName,
			UserID:              userID,
			DescriptionMarkdown: &descriptionMarkdown,
		}

		queryRegEx = `
			INSERT INTO teams \(name, display_name, user_id, info, created_at\)
			SELECT \$1::varchar, \$2, \$3, \$4, CURRENT_TIMESTAMP
			WHERE NOT EXISTS \(SELECT \* FROM teams WHERE name = \$1\)
			RETURNING id, name, display_name, user_id, info, team_logo_image, banner_image, background_image, created_at, updated_at
		`
	})

	Context("when insertion fails", func() {
		BeforeEach(func() {
			mock.ExpectQuery(queryRegEx).
				WithArgs(name, displayName, userID, nullString{&descriptionHTML}).
				WillReturnError(errors.New("db error"))
		})

		It("errors", func() {
			_, err := db.CreateTeam(context.Background(), params)
			Expect(err).To(HaveOccurred())

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})
	})

	Context("when insertion returns no rows", func() {
		BeforeEach(func() {
			mock.ExpectQuery(queryRegEx).
				WithArgs(name, displayName, userID, nullString{&descriptionHTML}).
				WillReturnRows(&sqlmock.Rows{})
		})

		It("errors", func() {
			_, err := db.CreateTeam(context.Background(), params)
			Expect(err).To(MatchError(ErrNoTeamCreated))

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})
	})

	Context("when insertion succeeds", func() {
		insertionTime := time.Now()

		BeforeEach(func() {
			teamRow := sqlmock.NewRows([]string{"id", "name", "display_name", "user_id", "info", "team_logo_image", "banner_image", "background_image", "created_at", "updated_at"}).
				AddRow(teamID, name, displayName, userID, descriptionHTML, nil, nil, nil, insertionTime, nil)

			mock.ExpectQuery(queryRegEx).
				WithArgs(name, displayName, userID, nullString{&descriptionHTML}).
				WillReturnRows(teamRow)
		})

		It("returns the inserted Team when insertion succeeds", func() {
			expectedTeam := Team{
				ID:                  teamID,
				Name:                name,
				DisplayName:         displayName,
				UserID:              userID,
				descriptionHTML:     descriptionHTML,
				descriptionMarkdown: descriptionMarkdown,
				Logo:                nil,
				Banner:              nil,
				Background:          nil,
				CreatedAt:           &insertionTime,
				UpdatedAt:           nil,
			}

			team, err := db.CreateTeam(context.Background(), params)
			Expect(err).NotTo(HaveOccurred())
			Expect(team).To(Equal(expectedTeam))

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})
	})
})
