package db

import (
	"context"
	"database/sql"
	"errors"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("DeleteFeaturedChannel", func() {
	var (
		db               *Client
		mock             sqlmock.Sqlmock
		teamID           string
		channelID        string
		deleteQueryRegEx string
		updateQueryRegEx string
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		postgresDB := &postgres.DB{
			DB: stub,
		}

		db = &Client{
			db: postgresDB,
			displayPositioner: &displayPositionHelper{
				db: postgresDB,
			},
		}

		teamID = "123"
		channelID = "999999999"
		deleteQueryRegEx = `DELETE FROM team_streams WHERE team_id = \$1 AND user_id = \$2`
		updateQueryRegEx = `UPDATE team_streams SET display_order = normalized.position - 1`
	})

	It("fails when beginning the transaction fails", func() {
		mock.ExpectBegin().WillReturnError(errors.New("transaction has not begun"))

		err := db.DeleteFeaturedChannel(context.Background(), teamID, channelID)
		Expect(err.Error()).To(ContainSubstring("db: failed to begin transaction for deleting row in team_streams: transaction has not begun"))

		mockError := mock.ExpectationsWereMet()
		Expect(mockError).NotTo(HaveOccurred())
	})

	Context("when transaction begins successfully", func() {
		BeforeEach(func() {
			mock.ExpectBegin()
		})

		It("errors when deleting featured channel fails", func() {
			mock.ExpectExec(deleteQueryRegEx).
				WithArgs(teamID, channelID).
				WillReturnError(errors.New("some error"))
			mock.ExpectRollback()

			err := db.DeleteFeaturedChannel(context.Background(), teamID, channelID)
			Expect(err).To(HaveOccurred())

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})

		It("errors when deleting featured channel returns no rows", func() {
			mock.ExpectExec(deleteQueryRegEx).
				WithArgs(teamID, channelID).
				WillReturnResult(sqlmock.NewResult(0, 0))
			mock.ExpectRollback()

			err := db.DeleteFeaturedChannel(context.Background(), teamID, channelID)
			Expect(err).To(Equal(ErrNoRowFoundForDeletion))

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})

		Context("when deleting featured channel succeeds", func() {
			BeforeEach(func() {
				mock.ExpectExec(deleteQueryRegEx).
					WithArgs(teamID, channelID).
					WillReturnResult(sqlmock.NewResult(0, 1))
			})

			It("errors when normalizing team featured channel display positions fails", func() {
				mock.ExpectExec(updateQueryRegEx).
					WithArgs(teamID).
					WillReturnError(errors.New("some error"))
				mock.ExpectRollback()

				err := db.DeleteFeaturedChannel(context.Background(), teamID, channelID)
				Expect(err).To(HaveOccurred())

				err = mock.ExpectationsWereMet()
				Expect(err).NotTo(HaveOccurred())
			})

			Context("when normalizing team featured channel display positions succeeds", func() {
				BeforeEach(func() {
					mock.ExpectExec(updateQueryRegEx).
						WithArgs(teamID).
						WillReturnResult(sqlmock.NewResult(0, 1000))
				})

				It("commits the transaction", func() {
					mock.ExpectCommit()

					err := db.DeleteFeaturedChannel(context.Background(), teamID, channelID)
					Expect(err).NotTo(HaveOccurred())

					err = mock.ExpectationsWereMet()
					Expect(err).NotTo(HaveOccurred())
				})
			})
		})
	})
})
