package db

import (
	"context"
	"database/sql"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// DeleteTeam transactionally deletes a team and all of its associated relations from DB.
func (c *Client) DeleteTeam(ctx context.Context, id string) error {
	tx, err := c.db.BeginTx(ctx, nil)
	if err != nil {
		return errors.Wrapf(err, "db: failed to begin transaction to delete team %s", id)
	}

	defer func() {
		if err != nil && err != sql.ErrTxDone {
			if err = tx.Rollback(); err != nil {
				log.WithError(err).WithField("team_id", id).
					Errorf("db: failed to rollback transaction for deleting team %s", id)
			}
		}
	}()

	if err = txDeleteTeamInvitations(ctx, tx, id); err != nil {
		return err
	}

	if err = txDeleteTeamFeaturedChannels(ctx, tx, id); err != nil {
		return err
	}

	if err = txDeleteTeamMemberships(ctx, tx, id); err != nil {
		return err
	}

	if err = txDeleteTeam(ctx, tx, id); err != nil {
		return err
	}

	if err = tx.Commit(); err != nil {
		return errors.Wrapf(err, "db: failed to commit transaction for deleting team %s", id)
	}

	return nil
}

func txDeleteTeamInvitations(ctx context.Context, tx *sql.Tx, id string) error {
	_, err := tx.ExecContext(ctx, "DELETE FROM team_invitations WHERE team_id = $1", id)
	if err != nil {
		return errors.Wrapf(err, "db: failed to delete team_invitations for team %s", id)
	}

	return nil
}

func txDeleteTeamFeaturedChannels(ctx context.Context, tx *sql.Tx, id string) error {
	_, err := tx.ExecContext(ctx, "DELETE FROM team_streams WHERE team_id = $1", id)
	if err != nil {
		return errors.Wrapf(err, "db: failed to delete team_streams for team %s", id)
	}

	return nil
}

func txDeleteTeamMemberships(ctx context.Context, tx *sql.Tx, id string) error {
	_, err := tx.ExecContext(ctx, "DELETE FROM team_users WHERE team_id = $1", id)
	if err != nil {
		return errors.Wrapf(err, "db: failed to delete team_users for team %s", id)
	}

	return nil
}

func txDeleteTeam(ctx context.Context, tx *sql.Tx, id string) error {
	result, err := tx.ExecContext(ctx, "DELETE FROM teams WHERE id = $1", id)
	if err != nil {
		return errors.Wrapf(err, "db: failed to delete teams where id = %s", id)
	}

	rowsAffected, err := result.RowsAffected()
	if err != nil || rowsAffected < 1 {
		return ErrNoTeamDeleted
	}

	return nil
}
