package db

import (
	"context"
	"database/sql"
	"errors"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("DeleteTeam", func() {
	var (
		db   *Client
		mock sqlmock.Sqlmock
		err  error

		teamID                     string
		invitationsQueryRegEx      string
		featuredChannelsQueryRegEx string
		membershipsQueryRegEx      string
		teamQueryRegEx             string
	)

	BeforeEach(func() {
		var stub *sql.DB

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		db = &Client{
			db: &postgres.DB{
				DB: stub,
			},
		}

		teamID = "123"
		invitationsQueryRegEx = `DELETE FROM team_invitations WHERE team_id = \$1`
		featuredChannelsQueryRegEx = `DELETE FROM team_streams WHERE team_id = \$1`
		membershipsQueryRegEx = `DELETE FROM team_users WHERE team_id = \$1`
		teamQueryRegEx = `DELETE FROM teams WHERE id = \$1`
	})

	AfterEach(func() {
		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	It("errors when beginning the transaction fails", func() {
		mock.ExpectBegin().WillReturnError(errors.New("transaction has not begun"))

		err = db.DeleteTeam(context.Background(), teamID)
		Expect(err.Error()).To(ContainSubstring("failed to begin transaction to delete team"))

		mockError := mock.ExpectationsWereMet()
		Expect(mockError).NotTo(HaveOccurred())
	})

	Context("when the transaction begins successfully", func() {
		BeforeEach(func() {
			mock.ExpectBegin()
		})

		It("errors when deleting the team's invitations fails", func() {
			mock.ExpectExec(invitationsQueryRegEx).WithArgs(teamID).WillReturnError(errors.New("some error"))
			mock.ExpectRollback()

			err = db.DeleteTeam(context.Background(), teamID)
			Expect(err).To(HaveOccurred())
			Expect(err.Error()).To(ContainSubstring("failed to delete team_invitations"))
		})

		Context("when deleting the team's invitations succeeds", func() {
			BeforeEach(func() {
				mock.ExpectExec(invitationsQueryRegEx).WithArgs(teamID).WillReturnResult(sqlmock.NewResult(0, 1))
			})

			It("errors when deleting the team's featured channels fails", func() {
				mock.ExpectExec(featuredChannelsQueryRegEx).WithArgs(teamID).WillReturnError(errors.New("some error"))
				mock.ExpectRollback()

				err = db.DeleteTeam(context.Background(), teamID)
				Expect(err).To(HaveOccurred())
				Expect(err.Error()).To(ContainSubstring("failed to delete team_streams"))
			})

			Context("when deleting the team's featured channels succeeds", func() {
				BeforeEach(func() {
					mock.ExpectExec(featuredChannelsQueryRegEx).WithArgs(teamID).WillReturnResult(sqlmock.NewResult(0, 1000))
				})

				It("errors when deleting the team's memberships fails", func() {
					mock.ExpectExec(membershipsQueryRegEx).WithArgs(teamID).WillReturnError(errors.New("some error"))
					mock.ExpectRollback()

					err = db.DeleteTeam(context.Background(), teamID)
					Expect(err).To(HaveOccurred())
					Expect(err.Error()).To(ContainSubstring("failed to delete team_users"))
				})

				Context("when deleting the team's memberships succeeds", func() {
					BeforeEach(func() {
						mock.ExpectExec(membershipsQueryRegEx).WithArgs(teamID).WillReturnResult(sqlmock.NewResult(0, 999))
					})

					It("errors when deleting the team fails", func() {
						mock.ExpectExec(teamQueryRegEx).WithArgs(teamID).WillReturnError(errors.New("some error"))
						mock.ExpectRollback()

						err = db.DeleteTeam(context.Background(), teamID)
						Expect(err).To(HaveOccurred())
						Expect(err.Error()).To(ContainSubstring("failed to delete teams where id = 123"))
					})

					It("errors when deleting the team results in no rows affected", func() {
						mock.ExpectExec(teamQueryRegEx).WithArgs(teamID).WillReturnResult(sqlmock.NewResult(0, 0))
						mock.ExpectRollback()

						err = db.DeleteTeam(context.Background(), teamID)
						Expect(err).To(MatchError(ErrNoTeamDeleted))
					})

					Context("when deleting the team succeeds", func() {
						BeforeEach(func() {
							mock.ExpectExec(teamQueryRegEx).WithArgs(teamID).WillReturnResult(sqlmock.NewResult(0, 1))
						})

						It("errors when committing the transaction fails", func() {
							mock.ExpectCommit().
								WillReturnError(errors.New("it's not you -- it's me. I'm just not really into commitments"))

							err = db.DeleteTeam(context.Background(), teamID)
							Expect(err).To(HaveOccurred())
							Expect(err.Error()).To(ContainSubstring("failed to commit transaction for deleting team"))
						})

						It("returns no error when the transaction commits", func() {
							mock.ExpectCommit()

							err = db.DeleteTeam(context.Background(), teamID)
							Expect(err).NotTo(HaveOccurred())
						})
					})
				})
			})
		})
	})
})
