package db

import (
	"context"
	"database/sql"
	"errors"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("something unique", func() {
	var (
		mock                   sqlmock.Sqlmock
		table                  string
		teamID                 string
		channelID              string
		membershipID           string
		currentDisplayPosition uint
		desiredDisplayPosition uint
		maxDisplayPosition     uint
		helper                 *displayPositionHelper
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		helper = &displayPositionHelper{
			db: &postgres.DB{
				DB: stub,
			},
		}

		table = "test_table"
		teamID = "123"
		channelID = "999999999"
		membershipID = "membership id"
		currentDisplayPosition = 5
		desiredDisplayPosition = 100000
		maxDisplayPosition = 9
	})

	It("fails when the beginning transaction fails", func() {
		mock.ExpectBegin().WillReturnError(errors.New("transaction has not begun"))

		err := helper.Update(context.Background(), table, teamID, channelID, desiredDisplayPosition)
		Expect(err.Error()).To(ContainSubstring("db: failed to begin transaction for updating row display position in test_table: transaction has not begun"))

		mockError := mock.ExpectationsWereMet()
		Expect(mockError).NotTo(HaveOccurred())
	})

	Context("when transaction begins successfully", func() {
		BeforeEach(func() {
			mock.ExpectBegin()
		})

		It("errors when normalizing team membership display positions fails", func() {
			mock.ExpectExec(`UPDATE test_table SET display_order = normalized.position - 1`).
				WithArgs(teamID).
				WillReturnError(errors.New("some error"))
			mock.ExpectRollback()

			err := helper.Update(context.Background(), table, teamID, channelID, desiredDisplayPosition)
			Expect(err.Error()).To(ContainSubstring("db: failed to update display positions for normalization in test_table: some error"))

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})

		Context("when normalizing team membership display positions succeeds", func() {
			BeforeEach(func() {
				mock.ExpectExec(`UPDATE test_table SET display_order = normalized.position - 1`).
					WithArgs(teamID).
					WillReturnResult(sqlmock.NewResult(0, 0))
			})

			It("errors when no membership exists", func() {
				mock.ExpectQuery(`SELECT id, display_order FROM test_table WHERE team_id = \$1 AND user_id = \$2`).
					WithArgs(teamID, channelID).
					WillReturnError(sql.ErrNoRows)
				mock.ExpectRollback()

				err := helper.Update(context.Background(), table, teamID, channelID, desiredDisplayPosition)
				Expect(err).To(Equal(sql.ErrNoRows))

				err = mock.ExpectationsWereMet()
				Expect(err).NotTo(HaveOccurred())
			})

			It("errors when selecting membership fails", func() {
				mock.ExpectQuery(`SELECT id, display_order FROM test_table WHERE team_id = \$1 AND user_id = \$2`).
					WithArgs(teamID, channelID).
					WillReturnError(errors.New("some error"))
				mock.ExpectRollback()

				err := helper.Update(context.Background(), table, teamID, channelID, desiredDisplayPosition)
				Expect(err.Error()).To(ContainSubstring("db: failed to scan row from test_table for display position: some error"))

				err = mock.ExpectationsWereMet()
				Expect(err).NotTo(HaveOccurred())
			})

			Context("when selecting membership succeeds", func() {
				BeforeEach(func() {
					mock.ExpectQuery(`SELECT id, display_order FROM test_table WHERE team_id = \$1 AND user_id = \$2`).
						WithArgs(teamID, channelID).
						WillReturnRows(sqlmock.NewRows([]string{"id", "display_order"}).AddRow(membershipID, currentDisplayPosition))
				})

				It("errors when selecting max display position for given team fails", func() {
					mock.ExpectQuery(`SELECT MAX\(display_order\) FROM test_table WHERE team_id = \$1`).
						WithArgs(teamID).
						WillReturnError(errors.New("some error"))
					mock.ExpectRollback()

					err := helper.Update(context.Background(), table, teamID, channelID, desiredDisplayPosition)
					Expect(err.Error()).To(ContainSubstring("db: failed to scan for maximum display order from test_table: some error"))

					err = mock.ExpectationsWereMet()
					Expect(err).NotTo(HaveOccurred())
				})

				Context("when selecting max display position for given team succeeds", func() {
					BeforeEach(func() {
						mock.ExpectQuery(`SELECT MAX\(display_order\) FROM test_table WHERE team_id = \$1`).
							WithArgs(teamID).
							WillReturnRows(sqlmock.NewRows([]string{"max"}).AddRow(maxDisplayPosition))
					})

					It("errors when updating given membership to out of max bound fails", func() {
						mock.ExpectExec(`UPDATE test_table SET display_order = \$1 WHERE id = \$2`).
							WithArgs(maxDisplayPosition+1, membershipID).
							WillReturnError(errors.New("some error"))
						mock.ExpectRollback()

						err := helper.Update(context.Background(), table, teamID, channelID, desiredDisplayPosition)
						Expect(err.Error()).To(ContainSubstring("db: failed to update row display position in test_table: some error"))

						err = mock.ExpectationsWereMet()
						Expect(err).NotTo(HaveOccurred())
					})

					Context("when updating given membership to out of max bound succeeds", func() {
						BeforeEach(func() {
							mock.ExpectExec(`UPDATE test_table SET display_order = \$1 WHERE id = \$2`).
								WithArgs(maxDisplayPosition+1, membershipID).
								WillReturnResult(sqlmock.NewResult(0, 0))
						})

						Context("when decreasing the given membership's position", func() {
							BeforeEach(func() {
								desiredDisplayPosition = currentDisplayPosition - 1
							})

							It("errors when incrementing other membership display positions fails", func() {
								mock.ExpectExec(`
										UPDATE test_table
										SET display_order = display_order \+ 1
										WHERE display_order >= \$1
										AND display_order < \$2
										AND team_id = \$3
									`).WithArgs(desiredDisplayPosition, currentDisplayPosition, teamID).WillReturnError(errors.New("some error"))
								mock.ExpectRollback()

								err := helper.Update(context.Background(), table, teamID, channelID, desiredDisplayPosition)
								Expect(err.Error()).To(ContainSubstring("db: failed to increment display positions of surrounding rows in test_table: some error"))

								err = mock.ExpectationsWereMet()
								Expect(err).NotTo(HaveOccurred())
							})
						})

						Context("when increasing the given membership's position", func() {
							BeforeEach(func() {
								desiredDisplayPosition = currentDisplayPosition + 1
							})

							It("errors when decrementing other membership display positions fails", func() {
								mock.ExpectExec(`
										UPDATE test_table
										SET display_order = display_order - 1
										WHERE display_order > \$1
										AND display_order <= \$2
										AND team_id = \$3
									`).WithArgs(currentDisplayPosition, desiredDisplayPosition, teamID).WillReturnError(errors.New("some error"))
								mock.ExpectRollback()

								err := helper.Update(context.Background(), table, teamID, channelID, desiredDisplayPosition)
								Expect(err.Error()).To(ContainSubstring("db: failed to decrement display positions of surrounding rows in test_table: some error"))

								err = mock.ExpectationsWereMet()
								Expect(err).NotTo(HaveOccurred())
							})

							Context("when updating other membership display positions succeeds", func() {
								BeforeEach(func() {
									mock.ExpectExec(`UPDATE test_table`).
										WithArgs(currentDisplayPosition, desiredDisplayPosition, teamID).
										WillReturnResult(sqlmock.NewResult(0, 0))
								})

								It("errors when updating given membership to desired position fails", func() {
									mock.ExpectExec(`UPDATE test_table SET display_order = \$1 WHERE id = \$2`).
										WithArgs(desiredDisplayPosition, membershipID).
										WillReturnError(errors.New("some error"))
									mock.ExpectRollback()

									err := helper.Update(context.Background(), table, teamID, channelID, desiredDisplayPosition)
									Expect(err.Error()).To(ContainSubstring("db: failed to update row display position in test_table: some error"))

									err = mock.ExpectationsWereMet()
									Expect(err).NotTo(HaveOccurred())
								})

								Context("when updating given membership to desired position succeeds", func() {
									BeforeEach(func() {
										mock.ExpectExec(`UPDATE test_table SET display_order = \$1 WHERE id = \$2`).
											WithArgs(desiredDisplayPosition, membershipID).
											WillReturnResult(sqlmock.NewResult(0, 0))
									})

									It("commits the transaction", func() {
										mock.ExpectCommit()

										err := helper.Update(context.Background(), table, teamID, channelID, desiredDisplayPosition)
										Expect(err).NotTo(HaveOccurred())

										err = mock.ExpectationsWereMet()
										Expect(err).NotTo(HaveOccurred())
									})
								})
							})
						})
					})
				})
			})
		})
	})
})
