package db

import (
	"context"
	"database/sql"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetChannelMemberships makes a SELECT query from DB
// for a list of records from the `team_users` table,
// given a particular channel ID (`team_users.user_id`). It
// joins this value with the `teams` table to return full team
// information also.
func (c *Client) GetChannelMemberships(ctx context.Context, channelID string) ([]Membership, error) {
	statement := `
		SELECT
			tu.user_id, tu.team_id, tu.view_revenue, tu.view_stats,
			t.id, t.name, t.display_name, t.user_id, t.info, t.team_logo_image, t.banner_image, t.background_image, t.created_at, t.updated_at
		FROM team_users AS tu
		JOIN teams AS t
		ON t.id = tu.team_id
		WHERE tu.user_id = $1
	`

	memberships := []Membership{}
	rows, err := c.db.QueryContext(ctx, statement, channelID)
	if err != nil {
		return nil, errors.Wrap(err, "db: failed to select from team_users and teams by channel id")
	}

	defer func() {
		if err = rows.Close(); err != nil {
			log.WithError(err).WithFields(log.Fields{
				"statement": statement,
				"user_id":   channelID,
			}).Error("db: failed to close rows")
		}
	}()

	for rows.Next() {
		nullableTeam := nullableTeam{}
		membership := Membership{}
		nullableRevenueRevealed := sql.NullBool{}
		nullableStatsRevealed := sql.NullBool{}

		err = rows.Scan(
			&membership.ChannelID,
			&membership.TeamID,
			&nullableRevenueRevealed,
			&nullableStatsRevealed,
			&nullableTeam.ID,
			&nullableTeam.Name,
			&nullableTeam.DisplayName,
			&nullableTeam.UserID,
			&nullableTeam.Description,
			&nullableTeam.LogoMetadata,
			&nullableTeam.BannerMetadata,
			&nullableTeam.BackgroundImageMetadata,
			&nullableTeam.CreatedAt,
			&nullableTeam.UpdatedAt,
		)

		team := nullableTeam.toTeam()
		membership.Team = &team

		if err != nil {
			return nil, errors.Wrap(err, "db: failed to scan team_users and teams rows")
		}

		membership.RevenueRevealed = nullableRevenueRevealed.Valid && nullableRevenueRevealed.Bool
		membership.StatsRevealed = nullableStatsRevealed.Valid && nullableStatsRevealed.Bool

		memberships = append(memberships, membership)
	}

	if err = rows.Err(); err != nil {
		return nil, errors.Wrap(err, "db: failed to iterate team_users and teams rows")
	}

	return memberships, nil
}
