package db

import (
	"context"
	"database/sql"
	"errors"
	"time"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("GetChannelMemberships", func() {
	var (
		db                   *Client
		mock                 sqlmock.Sqlmock
		channelID            string
		queryRegEx           string
		expectedMemberships  []Membership
		createdAt, updatedAt time.Time
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		db = &Client{
			db: &postgres.DB{
				DB: stub,
			},
		}

		channelID = "123"
		queryRegEx = `
			SELECT
				tu.user_id, tu.team_id, tu.view_revenue, tu.view_stats,
				t.id, t.name, t.display_name, t.user_id, t.info, t.team_logo_image, t.banner_image, t.background_image,
				t.created_at, t.updated_at
			FROM team_users AS tu
			JOIN teams AS t
			ON t.id = tu.team_id
			WHERE tu.user_id = \$1
		`

		logoID := "123"
		logoFormat := "jpeg"
		logoURL1 := "https://static-cdn.jtvnw.net/jtv_user_pictures/team-team1-team_logo_image-123-600x600.jpeg"
		logoURL2 := "https://static-cdn.jtvnw.net/jtv_user_pictures/team-team2-team_logo_image-123-600x600.jpeg"
		bannerID := "banneruid"
		bannerFormat := "jpeg"
		bannerURL1 := "https://static-cdn.jtvnw.net/jtv_user_pictures/team-team1-banner_image-banneruid-640x125.jpeg"
		bannerURL2 := "https://static-cdn.jtvnw.net/jtv_user_pictures/team-team2-banner_image-banneruid-640x125.jpeg"
		backgroundID := "backgrounduid"
		backgroundFormat := "bmp"
		backgroundURL1 := "https://static-cdn.jtvnw.net/jtv_user_pictures/team-team1-background_image-backgrounduid.bmp"
		backgroundURL2 := "https://static-cdn.jtvnw.net/jtv_user_pictures/team-team2-background_image-backgrounduid.bmp"
		createdAt = time.Date(2000, 1, 1, 0, 0, 0, 0, time.UTC)
		updatedAt = time.Date(3000, 12, 12, 1, 2, 3, 4, time.UTC)

		expectedMemberships = []Membership{
			{
				ChannelID:       channelID,
				TeamID:          "1",
				RevenueRevealed: true,
				StatsRevealed:   false,
				Team: &Team{
					ID:                  "1",
					Name:                "team1",
					DisplayName:         "Team One",
					descriptionHTML:     "<h1>Description1</h1>",
					descriptionMarkdown: "# Description1",
					UserID:              "owner",
					Logo: &Image{
						ID:     logoID,
						Format: logoFormat,
						URL:    logoURL1,
					},
					Banner: &Image{
						ID:     bannerID,
						Format: bannerFormat,
						URL:    bannerURL1,
					},
					Background: &Image{
						ID:     backgroundID,
						Format: backgroundFormat,
						URL:    backgroundURL1,
					},
					CreatedAt: &createdAt,
					UpdatedAt: &updatedAt,
				},
			},
			{
				ChannelID:       channelID,
				TeamID:          "2",
				RevenueRevealed: false,
				StatsRevealed:   true,
				Team: &Team{
					ID:                  "2",
					Name:                "team2",
					DisplayName:         "Team Two",
					descriptionHTML:     "<h2>Description2</h2>",
					descriptionMarkdown: "## Description2",
					UserID:              "owner2",
					Logo: &Image{
						ID:     logoID,
						Format: logoFormat,
						URL:    logoURL2,
					},
					Banner: &Image{
						ID:     bannerID,
						Format: bannerFormat,
						URL:    bannerURL2,
					},
					Background: &Image{
						ID:     backgroundID,
						Format: backgroundFormat,
						URL:    backgroundURL2,
					},
				},
			},
		}
	})

	It("errors when selecting memberships fails", func() {
		mock.ExpectQuery(queryRegEx).WithArgs(channelID).WillReturnError(errors.New("some error"))

		memberships, err := db.GetChannelMemberships(context.Background(), channelID)
		Expect(memberships).To(BeNil())
		Expect(err).To(HaveOccurred())

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	Context("when selecting memberships succeeds", func() {
		BeforeEach(func() {
			logoField := `
				:uid: 123
				:format: jpeg
			`
			bannerMetadata := `
				:uid: banneruid
				:format: jpeg
			`
			backgroundImageMetadata := `
				:uid: backgrounduid
				:format: bmp
			`
			twoRows := sqlmock.NewRows([]string{"user_id", "team_id", "view_revenue", "view_stats", "id", "name", "display_name", "user_id", "info", "team_logo_image", "banner_image", "background_image", "created_at", "updated_at"}).
				AddRow(channelID, "1", true, false, "1", "team1", "Team One", "owner", "<h1>Description1</h1>", logoField, bannerMetadata, backgroundImageMetadata, createdAt, updatedAt).
				AddRow(channelID, "2", false, true, "2", "team2", "Team Two", "owner2", "<h2>Description2</h2>", logoField, bannerMetadata, backgroundImageMetadata, nil, nil)

			mock.ExpectQuery(queryRegEx).WithArgs(channelID).WillReturnRows(twoRows)
		})

		It("returns memberships for given team ID", func() {
			memberships, err := db.GetChannelMemberships(context.Background(), channelID)

			Expect(memberships).To(Equal(expectedMemberships))
			Expect(err).NotTo(HaveOccurred())

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})
	})
})
