package db

import (
	"context"
	"database/sql"
	"errors"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("GetFeaturedChannel", func() {
	var (
		db         *Client
		mock       sqlmock.Sqlmock
		teamID     string
		channelID  string
		queryRegEx string
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		db = &Client{
			db: &postgres.DB{
				DB: stub,
			},
		}

		teamID = "123"
		channelID = "999999999"
		queryRegEx = `SELECT team_id, user_id FROM team_streams WHERE team_id = \$1 AND user_id = \$2 LIMIT 1`
	})

	It("errors when selecting featured channel fails", func() {
		emptyFeaturedChannel := FeaturedChannel{}
		mock.ExpectQuery(queryRegEx).WithArgs(teamID, channelID).WillReturnError(errors.New("some error"))

		featuredChannel, err := db.GetFeaturedChannel(context.Background(), teamID, channelID)
		Expect(featuredChannel).To(Equal(emptyFeaturedChannel))
		Expect(err).To(HaveOccurred())

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	It("errors when selecting featured channel returns no rows", func() {
		noRows := sqlmock.NewRows([]string{"team_id", "user_id"})
		emptyFeaturedChannel := FeaturedChannel{}
		mock.ExpectQuery(queryRegEx).WithArgs(teamID, channelID).WillReturnRows(noRows)

		featuredChannel, err := db.GetFeaturedChannel(context.Background(), teamID, channelID)
		Expect(featuredChannel).To(Equal(emptyFeaturedChannel))
		Expect(err).To(Equal(ErrNoFeaturedChannel))

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	Context("when selecting featured channel succeeds", func() {
		BeforeEach(func() {
			oneRow := sqlmock.NewRows([]string{"team_id", "user_id"}).AddRow(teamID, channelID)
			mock.ExpectQuery(queryRegEx).WithArgs(teamID, channelID).WillReturnRows(oneRow)
		})

		It("returns no error", func() {
			expectedFeaturedChannel := FeaturedChannel{
				ChannelID: channelID,
				TeamID:    teamID,
			}

			featuredChannel, err := db.GetFeaturedChannel(context.Background(), teamID, channelID)
			Expect(featuredChannel).To(Equal(expectedFeaturedChannel))
			Expect(err).NotTo(HaveOccurred())

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})
	})
})
