package db

import (
	"context"
	"database/sql"
	"errors"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("GetFeaturedChannels", func() {
	var (
		db         *Client
		mock       sqlmock.Sqlmock
		teamID     string
		queryRegEx string
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		db = &Client{
			db: &postgres.DB{
				DB: stub,
			},
		}

		teamID = "123"
		queryRegEx = `SELECT team_id, user_id FROM team_streams WHERE team_id = \$1 ORDER BY display_order ASC`
	})

	It("errors when selecting featured channel fails", func() {
		mock.ExpectQuery(queryRegEx).WithArgs(teamID).WillReturnError(errors.New("some error"))

		featuredChannels, err := db.GetFeaturedChannels(context.Background(), teamID)
		Expect(featuredChannels).To(BeNil())
		Expect(err).To(HaveOccurred())

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	Context("when selecting featured channel succeeds", func() {
		BeforeEach(func() {
			twoRows := sqlmock.NewRows([]string{"team_id", "user_id"}).
				AddRow(teamID, "1").
				AddRow(teamID, "2")
			mock.ExpectQuery(queryRegEx).WithArgs(teamID).WillReturnRows(twoRows)
		})

		It("returns featured channels for given team ID", func() {
			expectedFeaturedChannels := []FeaturedChannel{
				{
					ChannelID: "1",
					TeamID:    teamID,
				},
				{
					ChannelID: "2",
					TeamID:    teamID,
				},
			}

			featuredChannels, err := db.GetFeaturedChannels(context.Background(), teamID)
			Expect(featuredChannels).To(Equal(expectedFeaturedChannels))
			Expect(err).NotTo(HaveOccurred())

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})
	})
})
