package db

import (
	"context"
	"database/sql"
	"errors"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("GetInvitation", func() {
	var (
		db         *Client
		mock       sqlmock.Sqlmock
		teamID     string
		channelID  string
		queryRegEx string
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		db = &Client{
			db: &postgres.DB{
				DB: stub,
			},
		}

		teamID = "123"
		channelID = "999999999"
		queryRegEx = `SELECT team_id, user_id FROM team_invitations WHERE team_id = \$1 AND user_id = \$2 LIMIT 1`
	})

	It("errors when selecting invitation fails", func() {
		emptyInvitation := Invitation{}
		mock.ExpectQuery(queryRegEx).WithArgs(teamID, channelID).WillReturnError(errors.New("some error"))

		invitation, err := db.GetInvitation(context.Background(), teamID, channelID)
		Expect(invitation).To(Equal(emptyInvitation))
		Expect(err).To(HaveOccurred())

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	It("errors when selecting invitation returns no rows", func() {
		noRows := sqlmock.NewRows([]string{"team_id", "user_id"})
		emptyInvitation := Invitation{}
		mock.ExpectQuery(queryRegEx).WithArgs(teamID, channelID).WillReturnRows(noRows)

		invitation, err := db.GetInvitation(context.Background(), teamID, channelID)
		Expect(invitation).To(Equal(emptyInvitation))
		Expect(err).To(Equal(ErrNoInvitation))

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	Context("when selecting invitation succeeds", func() {
		oneRow := sqlmock.NewRows([]string{"team_id", "user_id"}).AddRow(teamID, channelID)
		expectedInvitation := Invitation{
			ChannelID: channelID,
			TeamID:    teamID,
		}
		BeforeEach(func() {
			mock.ExpectQuery(queryRegEx).WithArgs(teamID, channelID).WillReturnRows(oneRow)
		})

		It("returns no error", func() {
			invitation, err := db.GetInvitation(context.Background(), teamID, channelID)
			Expect(invitation).To(Equal(expectedInvitation))
			Expect(err).NotTo(HaveOccurred())

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})
	})
})
