package db

import (
	"context"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetInvitingTeamsForChannel makes a SELECT query from DB for a list of Teams from the
// `teams` table that have pending invitations in the `team_invitations` table for the given channel.
func (c *Client) GetInvitingTeamsForChannel(ctx context.Context, channelID string) ([]Team, error) {
	query := `
		SELECT
			t.id, t.name, t.display_name, t.user_id, t.info, t.team_logo_image, t.banner_image, t.background_image, t.created_at, t.updated_at
		FROM team_invitations AS i
		JOIN teams AS t
			ON t.id = i.team_id
		WHERE i.user_id = $1;
	`

	rows, err := c.db.QueryContext(ctx, query, channelID)
	if err != nil {
		return nil, errors.Wrap(err, "db: failed to query channel invitations")
	}

	defer func() {
		if err = rows.Close(); err != nil {
			log.WithError(err).Error("db: failed to close rows for channel invitations")
		}
	}()

	teams := []Team{}

	for rows.Next() {
		team, err := teamFromRows(rows)
		if err != nil {
			return nil, errors.Wrap(err, "db: failed to scan teams row for channel invitations")
		}

		teams = append(teams, team)
	}

	return teams, nil
}
