package db

import (
	"context"
	"database/sql"
	"errors"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("GetMembership", func() {
	var (
		db         *Client
		mock       sqlmock.Sqlmock
		teamID     string
		channelID  string
		queryRegEx string
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		db = &Client{
			db: &postgres.DB{
				DB: stub,
			},
		}

		teamID = "123"
		channelID = "999999999"
		queryRegEx = `SELECT team_id, user_id, view_revenue, view_stats FROM team_users WHERE team_id = \$1 AND user_id = \$2 LIMIT 1`
	})

	It("errors when selecting membership fails", func() {
		emptyMembership := Membership{}
		mock.ExpectQuery(queryRegEx).WithArgs(teamID, channelID).WillReturnError(errors.New("some error"))

		membership, err := db.GetMembership(context.Background(), teamID, channelID)
		Expect(membership).To(Equal(emptyMembership))
		Expect(err).To(HaveOccurred())

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	It("errors when selecting membership returns no rows", func() {
		noRows := sqlmock.NewRows([]string{"team_id", "user_id"})
		emptyMembership := Membership{}
		mock.ExpectQuery(queryRegEx).WithArgs(teamID, channelID).WillReturnRows(noRows)

		membership, err := db.GetMembership(context.Background(), teamID, channelID)
		Expect(membership).To(Equal(emptyMembership))
		Expect(err).To(Equal(ErrNoMembership))

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	Context("when selecting membership succeeds", func() {
		oneRow := sqlmock.NewRows([]string{"team_id", "user_id", "view_revenue", "view_stats"}).AddRow(teamID, channelID, true, true)
		expectedMembership := Membership{
			ChannelID:       channelID,
			TeamID:          teamID,
			RevenueRevealed: true,
			StatsRevealed:   true,
		}
		BeforeEach(func() {
			mock.ExpectQuery(queryRegEx).WithArgs(teamID, channelID).WillReturnRows(oneRow)
		})

		It("returns no error", func() {
			membership, err := db.GetMembership(context.Background(), teamID, channelID)
			Expect(membership).To(Equal(expectedMembership))
			Expect(err).NotTo(HaveOccurred())

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})
	})
})
