package db

import (
	"context"
	"database/sql"
	"errors"
	"time"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("GetTeamByID", func() {
	var (
		db     *Client
		mock   sqlmock.Sqlmock
		teamID string

		queryRegEx string
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		db = &Client{
			db: &postgres.DB{
				DB: stub,
			},
		}

		teamID = "123"
		queryRegEx = `
			SELECT id, name, display_name, user_id, info, team_logo_image, banner_image, background_image, created_at, updated_at
			FROM teams
			WHERE id = \$1
			LIMIT 1
		`
	})

	It("errors when selecting team fails", func() {
		emptyTeam := Team{}
		mock.ExpectQuery(queryRegEx).WithArgs(teamID).WillReturnError(errors.New("some error"))

		team, err := db.GetTeamByID(context.Background(), teamID)
		Expect(team).To(Equal(emptyTeam))
		Expect(err).To(HaveOccurred())

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	It("errors when selecting team returns no rows", func() {
		noRows := sqlmock.NewRows([]string{"id", "name", "user_id"})
		emptyTeam := Team{}
		mock.ExpectQuery(queryRegEx).WithArgs(teamID).WillReturnRows(noRows)

		team, err := db.GetTeamByID(context.Background(), teamID)
		Expect(team).To(Equal(emptyTeam))
		Expect(err).To(Equal(ErrNoTeam))

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	Context("when selecting team succeeds", func() {
		It("returns no error", func() {
			teamName := "Some Team"
			displayName := "da team"
			userID := "999999999"
			description := "<h2>Description</h2>"
			logoMetadata := `
				:uid: logouid
				:format: src
			`
			bannerMetadata := `
				:uid: banneruid
				:format: jpeg
			`
			backgroundImageMetadata := `
				:uid: backgrounduid
				:format: bmp
			`
			logoID := "logouid"
			logoFormat := "src"
			logoURL := "https://static-cdn.jtvnw.net/jtv_user_pictures/team-Some Team-team_logo_image-logouid-600x600.src"
			bannerID := "banneruid"
			bannerFormat := "jpeg"
			bannerURL := "https://static-cdn.jtvnw.net/jtv_user_pictures/team-Some Team-banner_image-banneruid-640x125.jpeg"
			backgroundID := "backgrounduid"
			backgroundFormat := "bmp"
			backgroundURL := "https://static-cdn.jtvnw.net/jtv_user_pictures/team-Some Team-background_image-backgrounduid.bmp"
			createdAt := time.Date(2000, 1, 1, 0, 0, 0, 0, time.UTC)
			updatedAt := time.Date(3000, 12, 12, 1, 2, 3, 4, time.UTC)

			oneRow := sqlmock.NewRows([]string{"id", "name", "display_name", "user_id", "info", "team_logo_image", "banner_image", "background_image", "created_at", "updated_at"}).
				AddRow(teamID, teamName, displayName, userID, description, logoMetadata, bannerMetadata, backgroundImageMetadata, createdAt, updatedAt)

			expectedTeam := Team{
				ID:                  teamID,
				Name:                teamName,
				DisplayName:         displayName,
				UserID:              userID,
				descriptionHTML:     description,
				descriptionMarkdown: "## Description",
				Logo: &Image{
					ID:     logoID,
					Format: logoFormat,
					URL:    logoURL,
				},
				Banner: &Image{
					ID:     bannerID,
					Format: bannerFormat,
					URL:    bannerURL,
				},
				Background: &Image{
					ID:     backgroundID,
					Format: backgroundFormat,
					URL:    backgroundURL,
				},
				CreatedAt: &createdAt,
				UpdatedAt: &updatedAt,
			}

			mock.ExpectQuery(queryRegEx).WithArgs(teamID).WillReturnRows(oneRow)

			team, err := db.GetTeamByID(context.Background(), teamID)
			Expect(team).To(Equal(expectedTeam))
			Expect(err).NotTo(HaveOccurred())

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})
	})
})
