package db

import (
	"context"
	"database/sql"
	"fmt"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetTeamMemberships makes a SELECT query from DB
// for a list of records from the `team_users` table,
// given a particular team ID (`team_users.team_id`).
func (c *Client) GetTeamMemberships(ctx context.Context, teamID string, filter *MembershipsFilter) ([]Membership, error) {
	statement := `
		SELECT team_id, user_id, view_revenue, view_stats
		FROM team_users
		WHERE team_id = $1
	`

	if filter != nil {
		if filter.RevenueRevealed != nil {
			statement += fmt.Sprintf(" AND view_revenue = %t", *filter.RevenueRevealed)
		}

		if filter.StatsRevealed != nil {
			statement += fmt.Sprintf(" AND view_stats = %t", *filter.StatsRevealed)
		}
	}

	statement += " ORDER BY display_order ASC"

	memberships := []Membership{}
	rows, err := c.db.QueryContext(ctx, statement, teamID)
	if err != nil {
		return nil, errors.Wrap(err, "db: failed to select team_users by team_id")
	}

	defer func() {
		if err = rows.Close(); err != nil {
			log.WithError(err).WithFields(log.Fields{
				"statement": statement,
				"team_id":   teamID,
			}).Error("db: failed to close rows")
		}
	}()

	for rows.Next() {
		membership := Membership{}
		nullableRevenueRevealed := sql.NullBool{}
		nullableStatsRevealed := sql.NullBool{}

		err = rows.Scan(
			&membership.TeamID,
			&membership.ChannelID,
			&nullableRevenueRevealed,
			&nullableStatsRevealed,
		)

		if err != nil {
			return nil, errors.Wrap(err, "db: failed to scan team_users rows")
		}

		membership.RevenueRevealed = nullableRevenueRevealed.Valid && nullableRevenueRevealed.Bool
		membership.StatsRevealed = nullableStatsRevealed.Valid && nullableStatsRevealed.Bool

		memberships = append(memberships, membership)
	}

	if err = rows.Err(); err != nil {
		return nil, errors.Wrap(err, "db: failed to iterate team_users rows")
	}

	return memberships, nil
}
