package db

import (
	"context"
	"database/sql"
	"errors"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("GetTeamMemberships", func() {
	var (
		db                  *Client
		mock                sqlmock.Sqlmock
		teamID              string
		queryRegEx          string
		expectedMemberships []Membership
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		db = &Client{
			db: &postgres.DB{
				DB: stub,
			},
		}

		teamID = "123"
		queryRegEx = `SELECT team_id, user_id, view_revenue, view_stats FROM team_users WHERE team_id = \$1`
		expectedMemberships = []Membership{
			{
				ChannelID:       "1",
				TeamID:          teamID,
				RevenueRevealed: true,
				StatsRevealed:   false,
			},
			{
				ChannelID:       "2",
				TeamID:          teamID,
				RevenueRevealed: false,
				StatsRevealed:   true,
			},
		}
	})

	It("errors when selecting invitation fails", func() {
		mock.ExpectQuery(queryRegEx).WithArgs(teamID).WillReturnError(errors.New("some error"))

		invitations, err := db.GetTeamMemberships(context.Background(), teamID, nil)
		Expect(invitations).To(BeNil())
		Expect(err).To(HaveOccurred())

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	Context("when selecting invitation succeeds", func() {
		Context("with no filters", func() {
			BeforeEach(func() {
				twoRows := sqlmock.NewRows([]string{"team_id", "user_id", "view_revenue", "view_stats"}).
					AddRow(teamID, "1", true, false).
					AddRow(teamID, "2", false, true)
				mock.ExpectQuery(queryRegEx).WithArgs(teamID).WillReturnRows(twoRows)
			})

			It("returns invitations for given team ID", func() {
				invitations, err := db.GetTeamMemberships(context.Background(), teamID, nil)
				Expect(invitations).To(Equal(expectedMemberships))
				Expect(err).NotTo(HaveOccurred())

				err = mock.ExpectationsWereMet()
				Expect(err).NotTo(HaveOccurred())
			})
		})

		Context("with stats revealed", func() {
			BeforeEach(func() {
				oneRow := sqlmock.NewRows([]string{"team_id", "user_id", "view_revenue", "view_stats"}).
					AddRow(teamID, "1", true, false)
				mock.ExpectQuery(`SELECT team_id, user_id, view_revenue, view_stats FROM team_users WHERE team_id = \$1 AND view_revenue = true`).
					WithArgs(teamID).
					WillReturnRows(oneRow)
			})

			It("returns invitations for given team ID with stats revealed", func() {
				truthy := true
				invitations, err := db.GetTeamMemberships(context.Background(), teamID, &MembershipsFilter{RevenueRevealed: &truthy})
				Expect(invitations).To(Equal([]Membership{expectedMemberships[0]}))
				Expect(err).NotTo(HaveOccurred())

				err = mock.ExpectationsWereMet()
				Expect(err).NotTo(HaveOccurred())
			})
		})

		Context("with revenue revealed", func() {
			BeforeEach(func() {
				oneRow := sqlmock.NewRows([]string{"team_id", "user_id", "view_revenue", "view_stats"}).
					AddRow(teamID, "2", false, true)
				mock.ExpectQuery(`SELECT team_id, user_id, view_revenue, view_stats FROM team_users WHERE team_id = \$1 AND view_stats = true`).
					WithArgs(teamID).
					WillReturnRows(oneRow)
			})

			It("returns invitations for given team ID with stats revealed", func() {
				truthy := true
				invitations, err := db.GetTeamMemberships(context.Background(), teamID, &MembershipsFilter{StatsRevealed: &truthy})
				Expect(invitations).To(Equal([]Membership{expectedMemberships[1]}))
				Expect(err).NotTo(HaveOccurred())

				err = mock.ExpectationsWereMet()
				Expect(err).NotTo(HaveOccurred())
			})
		})
	})
})
