package db

import (
	"context"
	"fmt"
	"strings"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

const (
	// GetTeamsMaxLimit is the maximum number of selected teams.
	GetTeamsMaxLimit = 100

	// GetTeamsDefaultLimit is the default number of selected teams.
	GetTeamsDefaultLimit = GetTeamsMaxLimit
)

// GetTeams makes a SELECT query from DB for records
// from the `teams` table matching the given filter(s).
func (c *Client) GetTeams(ctx context.Context, filter TeamsFilter) ([]Team, error) {
	direction := OrderByDirectionAsc
	if strings.ToUpper(filter.OrderByDirection) == OrderByDirectionDesc {
		direction = OrderByDirectionDesc
	}

	statement := fmt.Sprintf(`
		SELECT id, name, display_name, user_id, info, team_logo_image, banner_image, background_image, created_at, updated_at
		FROM teams
		WHERE ($1 = '' OR name = $1)
		AND ($2 = '' OR user_id::text = $2)
		ORDER BY id %s
		LIMIT $3
		OFFSET $4
	`, direction)

	limit := uint(GetTeamsDefaultLimit)
	if filter.Limit != nil && *filter.Limit <= GetTeamsMaxLimit {
		limit = *filter.Limit
	}

	teams := []Team{}

	rows, err := c.db.QueryContext(ctx, statement, filter.Name, filter.UserID, limit, filter.Offset)
	if err != nil {
		return teams, errors.Wrap(err, "db: failed to query teams")
	}

	defer func() {
		if err = rows.Close(); err != nil {
			log.WithError(err).Error("db: failed to close rows for teams")
		}
	}()

	for rows.Next() {
		team, err := teamFromRows(rows)
		if err != nil {
			return teams, errors.Wrap(err, "db: failed to scan team")
		}

		teams = append(teams, team)
	}

	return teams, nil
}
