package db

import (
	"context"
	"database/sql"
	"errors"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("GetTeamsCount", func() {
	var (
		db         *Client
		mock       sqlmock.Sqlmock
		filter     TeamsFilter
		queryRegEx string
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		db = &Client{
			db: &postgres.DB{
				DB: stub,
			},
		}

		filter = TeamsFilter{
			Name: "staff",
		}

		queryRegEx = `SELECT COUNT\(\*\) FROM teams WHERE \$1 = '' OR name = \$1`
	})

	It("errors when selecting team count fails", func() {
		mock.ExpectQuery(queryRegEx).WithArgs().WillReturnError(errors.New("some error"))

		count, err := db.GetTeamsCount(context.Background(), filter)
		Expect(err).To(HaveOccurred())
		Expect(count).To(Equal(uint(0)))

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	Context("when selecting team count succeeds", func() {
		It("returns no error", func() {
			mock.ExpectQuery(queryRegEx).WillReturnRows(sqlmock.NewRows([]string{"count"}).AddRow(123))

			count, err := db.GetTeamsCount(context.Background(), filter)
			Expect(err).NotTo(HaveOccurred())
			Expect(count).To(Equal(uint(123)))

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})
	})
})
