package db

import (
	"context"
	"database/sql"
	"errors"
	"strings"
	"time"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("GetTeams", func() {
	var (
		db         *Client
		mock       sqlmock.Sqlmock
		queryRegEx string
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		db = &Client{
			db: &postgres.DB{
				DB: stub,
			},
		}

		queryRegEx = `
			SELECT id, name, display_name, user_id, info, team_logo_image, banner_image, background_image, created_at, updated_at
			FROM teams
			WHERE \(\$1 = '' OR name = \$1\)
				AND \(\$2 = '' OR user_id::text = \$2\)
			ORDER BY id ASC
			LIMIT \$3
			OFFSET \$4
		`
	})

	Context("with no filter", func() {
		var filter TeamsFilter

		BeforeEach(func() {
			filter = TeamsFilter{}
		})

		It("errors when selecting team fails", func() {
			mock.ExpectQuery(queryRegEx).
				WithArgs("", "", GetTeamsDefaultLimit, 0).
				WillReturnError(errors.New("some error"))

			teams, err := db.GetTeams(context.Background(), filter)
			Expect(err).To(HaveOccurred())
			Expect(teams).To(HaveLen(0))

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})

		It("errors when selecting team returns no rows", func() {
			noRows := sqlmock.NewRows([]string{})
			mock.ExpectQuery(queryRegEx).
				WithArgs("", "", GetTeamsDefaultLimit, 0).
				WillReturnRows(noRows)

			teams, err := db.GetTeams(context.Background(), filter)
			Expect(err).NotTo(HaveOccurred())
			Expect(teams).To(HaveLen(0))

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})

		Context("when selecting team succeeds", func() {
			It("returns no error", func() {
				teamID := "123"
				teamName := "staff"
				displayName := "da team"
				userID := "999999999"
				maliciousDescription := `<a href="http://trusted.org/search.cgi?val=<SCRIPT SRC='http://evil.org/badkama.js'></SCRIPT>">Click here!</a>`
				logoMetadata := `
					:uid: logouid
					:format: src
				`
				bannerMetadata := `
					:uid: banneruid
					:format: jpeg
				`
				backgroundImageMetadata := `
					:uid: backgrounduid
					:format: bmp
				`
				logoID := "logouid"
				logoFormat := "src"
				logoURL := "https://static-cdn.jtvnw.net/jtv_user_pictures/team-staff-team_logo_image-logouid-600x600.src"
				bannerID := "banneruid"
				bannerFormat := "jpeg"
				bannerURL := "https://static-cdn.jtvnw.net/jtv_user_pictures/team-staff-banner_image-banneruid-640x125.jpeg"
				backgroundID := "backgrounduid"
				backgroundFormat := "bmp"
				backgroundURL := "https://static-cdn.jtvnw.net/jtv_user_pictures/team-staff-background_image-backgrounduid.bmp"
				createdAt := time.Date(2000, 1, 1, 0, 0, 0, 0, time.UTC)
				updatedAt := time.Date(3000, 12, 12, 1, 2, 3, 4, time.UTC)

				oneRow := sqlmock.NewRows([]string{"id", "name", "display_name", "user_id", "info", "team_logo_image", "banner_image", "background_image", "created_at", "updated_at"}).
					AddRow(teamID, teamName, displayName, userID, maliciousDescription, logoMetadata, bannerMetadata, backgroundImageMetadata, createdAt, updatedAt)

				expectedTeam := Team{
					ID:                  teamID,
					Name:                teamName,
					DisplayName:         displayName,
					UserID:              userID,
					descriptionHTML:     maliciousDescription, // assume that DB's `teams.info` is sanitized
					descriptionMarkdown: "Click here!",
					Logo: &Image{
						ID:     logoID,
						Format: logoFormat,
						URL:    logoURL,
					},
					Banner: &Image{
						ID:     bannerID,
						Format: bannerFormat,
						URL:    bannerURL,
					},
					Background: &Image{
						ID:     backgroundID,
						Format: backgroundFormat,
						URL:    backgroundURL,
					},
					CreatedAt: &createdAt,
					UpdatedAt: &updatedAt,
				}

				mock.ExpectQuery(queryRegEx).
					WithArgs("", "", GetTeamsDefaultLimit, 0).
					WillReturnRows(oneRow)

				teams, err := db.GetTeams(context.Background(), filter)
				Expect(err).NotTo(HaveOccurred())

				Expect(teams).To(HaveLen(1))
				Expect(teams[0]).To(Equal(expectedTeam))

				err = mock.ExpectationsWereMet()
				Expect(err).NotTo(HaveOccurred())
			})
		})
	})

	Context("with filter", func() {
		var filter TeamsFilter

		BeforeEach(func() {
			limit := uint(101)
			filter = TeamsFilter{
				Limit:            &limit,
				Name:             "staff",
				Offset:           2,
				OrderByDirection: OrderByDirectionDesc,
				UserID:           "123",
			}
		})

		It("returns no error", func() {
			mock.ExpectQuery(strings.Replace(queryRegEx, "ASC", "DESC", -1)).
				WithArgs(filter.Name, filter.UserID, GetTeamsDefaultLimit, 2).
				WillReturnRows(&sqlmock.Rows{})

			teams, err := db.GetTeams(context.Background(), filter)
			Expect(err).NotTo(HaveOccurred())

			Expect(teams).To(HaveLen(0))

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})
	})
})
