package db

import (
	"fmt"
	"time"

	"code.justin.tv/cb/roster/internal/description"
	"code.justin.tv/cb/roster/internal/image"
)

// Team belongs to an owner via UserID.
type Team struct {
	ID          string
	Name        string
	UserID      string
	DisplayName string

	descriptionHTML     string
	descriptionMarkdown string

	Logo       *Image
	Banner     *Image
	Background *Image

	CreatedAt *time.Time
	UpdatedAt *time.Time
}

// DescriptionHTML is the public getter method
// for a Team's description in HTML format.
func (t *Team) DescriptionHTML() string {
	return t.descriptionHTML
}

// DescriptionMarkdown is the public getter method
// for a Team's description in Markdown format.
func (t *Team) DescriptionMarkdown() string {
	return t.descriptionMarkdown
}

// SetDescriptionWithMarkdown converts Markdown to HTML,
// allowing a Team to have description in both formats.
func (t *Team) SetDescriptionWithMarkdown(markdown string) {
	t.descriptionHTML = description.MarkdownToHTML(markdown)
	t.descriptionMarkdown = markdown
}

// SetDescriptionWithHTML converts HTML to Markdown,
// allowing a Team to have description in both formats.
//
// We assume that a team's description (`teams.info`)
// from DB has already been sanitized.
func (t *Team) SetDescriptionWithHTML(html string) {
	t.descriptionHTML = html
	t.descriptionMarkdown = description.HTMLToMarkdown(html)
}

// SetLogo sets the Logo of a Team, setting all fields of the underlying Image.
func (t *Team) SetLogo(id, format string) {
	if t.Logo == nil {
		t.Logo = &Image{}
	}

	t.Logo.ID = id
	t.Logo.Format = format
	t.Logo.URL = image.LegacyLogoURL(t.Name, id, format)
}

// SetBanner sets the Banner of a Team, setting all fields of the underlying Image.
func (t *Team) SetBanner(id, format string) {
	if t.Banner == nil {
		t.Banner = &Image{}
	}

	t.Banner.ID = id
	t.Banner.Format = format
	t.Banner.URL = image.LegacyBannerURL(t.Name, id, format)
}

// SetBackground sets the Background of a Team, setting all fields of the underlying Image.
func (t *Team) SetBackground(id, format string) {
	if t.Background == nil {
		t.Background = &Image{}
	}

	t.Background.ID = id
	t.Background.Format = format
	t.Background.URL = image.LegacyBackgroundURL(t.Name, id, format)
}

// Image belongs to a Team.
type Image struct {
	ID     string
	Format string
	URL    string
}

func (i *Image) dbString() *string {
	// web/web Rails inserts more metadata into the image columns than necessary --
	// these two attributes are not used in any way:
	// - :sizes:
	// - :ratio:
	//
	// We do not replicate the behavior here --
	// we only keep the :uid: and :format: properties, when available.
	if i.ID == "" || i.Format == "" {
		return nil
	}

	dbString := fmt.Sprintf(":uid: %s\n:format: %s", i.ID, i.Format)

	return &dbString
}

// ORDER BY direction can only be ASC or DESC.
const (
	OrderByDirectionAsc  = "ASC"
	OrderByDirectionDesc = "DESC"
)

// TeamsFilter contains optional fields to filter
// the selection of Teams.
type TeamsFilter struct {
	Limit            *uint
	Name             string
	Offset           uint
	OrderByDirection string
	UserID           string
}

// Membership ties a ChannelID to a Team via TeamID.
type Membership struct {
	ChannelID       string
	TeamID          string
	RevenueRevealed bool
	StatsRevealed   bool
	Team            *Team
}

// MembershipsFilter contains optional fields to
// filter the selection of Memberships.
type MembershipsFilter struct {
	RevenueRevealed *bool
	StatsRevealed   *bool
}

// TeamChannelJoin ties a ChannelID to a Team via TeamID
type TeamChannelJoin struct {
	ChannelID string
	TeamID    string
}

// Invitation ties a ChannelID to a Team via TeamID.
type Invitation TeamChannelJoin

// FeaturedChannel ties a ChannelID to a Team via TeamID.
type FeaturedChannel TeamChannelJoin
