package db

import (
	"context"
	"database/sql"
	"fmt"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

func (c *Client) deleteRowWithDisplayOrder(ctx context.Context, table, teamID, channelID string) error {
	tx, err := c.db.BeginTx(ctx, nil)
	if err != nil {
		return errors.Wrap(err, fmt.Sprintf("db: failed to begin transaction for deleting row in %s", table))
	}

	defer func() {
		if err == nil {
			return
		}

		if err = tx.Rollback(); err != nil {
			log.WithError(err).WithFields(log.Fields{
				"team_id":    teamID,
				"channel_id": channelID,
			}).Error(fmt.Sprintf("db: failed to rollback transaction for deleting row in %s", table))
		}
	}()

	if err = txDeleteRow(ctx, tx, table, teamID, channelID); err != nil {
		return err
	}

	if err = c.displayPositioner.Normalize(ctx, tx, table, teamID); err != nil {
		return err
	}

	if err = tx.Commit(); err != nil {
		return errors.Wrap(err, fmt.Sprintf("db: failed to commit transaction for deleting row in %s", table))
	}

	return nil
}

func txDeleteRow(ctx context.Context, tx *sql.Tx, table, teamID, channelID string) error {
	statement := fmt.Sprintf("DELETE FROM %s WHERE team_id = $1 AND user_id = $2", table)

	result, err := tx.ExecContext(ctx, statement, teamID, channelID)
	if err != nil {
		return errors.Wrap(err, fmt.Sprintf("db: failed to delete row from %s", table))
	}

	rowsAffected, err := result.RowsAffected()
	if err != nil || rowsAffected < 1 {
		return ErrNoRowFoundForDeletion
	}

	return nil
}

func (c *Client) getRow(ctx context.Context, table, teamID, channelID string) (TeamChannelJoin, error) {
	statement := fmt.Sprintf(`
		SELECT team_id, user_id
		FROM %s
		WHERE team_id = $1 AND user_id = $2
		LIMIT 1
	`, table)

	result := TeamChannelJoin{}
	row := c.db.QueryRowContext(ctx, statement, teamID, channelID)
	err := row.Scan(&result.TeamID, &result.ChannelID)

	return result, err
}

func (c *Client) getRows(ctx context.Context, table, orderBy, teamID string) ([]TeamChannelJoin, error) {
	statement := fmt.Sprintf(`
		SELECT team_id, user_id
		FROM %s
		WHERE team_id = $1
		ORDER BY %s ASC
	`, table, orderBy)

	results := []TeamChannelJoin{}
	rows, err := c.db.QueryContext(ctx, statement, teamID)
	if err != nil {
		return nil, errors.Wrap(err, fmt.Sprintf("db: failed to select from %s by team_id", table))
	}

	defer func() {
		if err = rows.Close(); err != nil {
			log.WithError(err).WithFields(log.Fields{
				"statement": statement,
				"team_id":   teamID,
			}).Error("db: failed to close rows")
		}
	}()

	for rows.Next() {
		result := TeamChannelJoin{}

		if err = rows.Scan(&result.TeamID, &result.ChannelID); err != nil {
			return nil, errors.Wrap(err, fmt.Sprintf("db: failed to scan %s rows", table))
		}

		results = append(results, result)
	}

	if err = rows.Err(); err != nil {
		return nil, errors.Wrap(err, fmt.Sprintf("db: failed to iterate %s rows", table))
	}

	return results, nil
}
