package db

import (
	"context"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// SearchTeamsFilter is for Client.SearchTeams.
type SearchTeamsFilter struct {
	Limit  uint
	Name   string
	Offset uint
}

// SearchTeams makes a SELECT query from DB for records
// from the `teams` table matching the given filter(s).
func (c *Client) SearchTeams(ctx context.Context, filter SearchTeamsFilter) ([]Team, error) {
	statement := `
		SELECT id, name, display_name, user_id, info, team_logo_image, banner_image, background_image, created_at, updated_at
		FROM teams
		WHERE name ILIKE '%' || $1 || '%'
		LIMIT $2
		OFFSET $3
	`

	teams := []Team{}

	rows, err := c.db.QueryContext(ctx, statement, filter.Name, filter.Limit, filter.Offset)
	if err != nil {
		return nil, errors.Wrap(err, "db: failed to query teams")
	}

	defer func() {
		if err = rows.Close(); err != nil {
			log.WithError(err).Error("db: failed to close rows for teams")
		}
	}()

	for rows.Next() {
		team, err := teamFromRows(rows)
		if err != nil {
			return nil, errors.Wrap(err, "db: failed to scan team")
		}

		teams = append(teams, team)
	}

	return teams, nil
}
