package db

import (
	"context"
	"database/sql"
	"errors"
	"time"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("SearchTeams", func() {
	var (
		db   *Client
		mock sqlmock.Sqlmock

		filter     SearchTeamsFilter
		queryRegEx string
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		db = &Client{
			db: &postgres.DB{
				DB: stub,
			},
		}

		filter = SearchTeamsFilter{
			Limit:  uint(101),
			Name:   "staff",
			Offset: 2,
		}

		queryRegEx = `
			SELECT id, name, display_name, user_id, info, team_logo_image, banner_image, background_image, created_at, updated_at
			FROM teams
			WHERE name ILIKE '%' || \$1 || '%'
			LIMIT \$2
			OFFSET \$3
		`
	})

	It("errors when selecting team fails", func() {
		mock.ExpectQuery(queryRegEx).WithArgs("staff", 101, 2).WillReturnError(errors.New("some error"))

		teams, err := db.SearchTeams(context.Background(), filter)
		Expect(err).To(HaveOccurred())
		Expect(teams).To(HaveLen(0))

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	It("errors when selecting team returns no rows", func() {
		mock.ExpectQuery(queryRegEx).WithArgs("staff", 101, 2).WillReturnRows(sqlmock.NewRows([]string{}))

		teams, err := db.SearchTeams(context.Background(), filter)
		Expect(err).NotTo(HaveOccurred())
		Expect(teams).To(HaveLen(0))

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	It("returns teams when there are no errors", func() {
		teamID := "123"
		teamName := "staff"
		displayName := "da team"
		userID := "999999999"
		description := "description"

		logoMetadata := `
			:uid: logouid
			:format: src
		`
		bannerMetadata := `
			:uid: banneruid
			:format: jpeg
		`
		backgroundImageMetadata := `
			:uid: backgrounduid
			:format: bmp
		`

		createdAt := time.Date(2000, 1, 1, 0, 0, 0, 0, time.UTC)
		updatedAt := time.Date(3000, 12, 12, 1, 2, 3, 4, time.UTC)

		oneRow := sqlmock.NewRows([]string{"id", "name", "display_name", "user_id", "info", "team_logo_image", "banner_image", "background_image", "created_at", "updated_at"}).
			AddRow(teamID, teamName, displayName, userID, description, logoMetadata, bannerMetadata, backgroundImageMetadata, createdAt, updatedAt)

		mock.ExpectQuery(queryRegEx).WithArgs("staff", 101, 2).WillReturnRows(oneRow)

		teams, err := db.SearchTeams(context.Background(), filter)
		Expect(err).NotTo(HaveOccurred())

		Expect(teams).To(HaveLen(1))

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})
})
