package db

import (
	"database/sql"
	"regexp"

	"github.com/lib/pq"
	"github.com/pkg/errors"
)

// nullableTeam which is able to take image metadata (on db load)
type nullableTeam struct {
	ID                      string
	Name                    sql.NullString
	DisplayName             sql.NullString
	Description             sql.NullString
	UserID                  sql.NullString
	LogoMetadata            sql.NullString
	BannerMetadata          sql.NullString
	BackgroundImageMetadata sql.NullString
	CreatedAt               pq.NullTime
	UpdatedAt               pq.NullTime
}

// toTeam returns a correctly populated Team object
func (n *nullableTeam) toTeam() Team {
	team := Team{
		ID:          n.ID,
		Name:        n.Name.String,
		UserID:      n.UserID.String,
		DisplayName: n.DisplayName.String,
	}

	team.SetDescriptionWithHTML(n.Description.String)

	if id := getTeamImageUID(n.LogoMetadata.String); id != "" {
		if format := getTeamImageFormat(n.LogoMetadata.String); format != "" {
			team.SetLogo(id, format)
		}
	}

	if id := getTeamImageUID(n.BannerMetadata.String); id != "" {
		if format := getTeamImageFormat(n.BannerMetadata.String); format != "" {
			team.SetBanner(id, format)
		}
	}

	if id := getTeamImageUID(n.BackgroundImageMetadata.String); id != "" {
		if format := getTeamImageFormat(n.BackgroundImageMetadata.String); format != "" {
			team.SetBackground(id, format)
		}
	}

	if n.CreatedAt.Valid {
		team.CreatedAt = &n.CreatedAt.Time
	}

	if n.UpdatedAt.Valid {
		team.UpdatedAt = &n.UpdatedAt.Time
	}

	return team
}

func teamFromRows(rows *sql.Rows) (Team, error) {
	team := nullableTeam{}

	err := rows.Scan(
		&team.ID,
		&team.Name,
		&team.DisplayName,
		&team.UserID,
		&team.Description,
		&team.LogoMetadata,
		&team.BannerMetadata,
		&team.BackgroundImageMetadata,
		&team.CreatedAt,
		&team.UpdatedAt,
	)

	if err != nil {
		return Team{}, errors.Wrap(err, "db: failed to scan teams row")
	}

	return team.toTeam(), nil
}

var (
	teamImageUIDRegex    = regexp.MustCompile(":uid:[[:blank:]]+([[:alnum:]]+)")
	teamImageFormatRegex = regexp.MustCompile(":format:[[:blank:]]+([[:alnum:]]+)")
)

func getTeamImageUID(metadata string) string {
	matches := teamImageUIDRegex.FindStringSubmatch(metadata)

	if len(matches) != 2 {
		return ""
	}

	return matches[1]
}

func getTeamImageFormat(metadata string) string {
	matches := teamImageFormatRegex.FindStringSubmatch(metadata)

	if len(matches) != 2 {
		return ""
	}

	return matches[1]
}
