package db

import (
	"context"
	"database/sql"
	"errors"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("UpdateMembership", func() {
	var (
		db   *Client
		mock sqlmock.Sqlmock

		teamID          string
		channelID       string
		revenueRevealed bool
		statsRevealed   bool

		commandRegex string
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		db = &Client{
			db: &postgres.DB{
				DB: stub,
			},
		}

		teamID = "123"
		channelID = "999999999"
		revenueRevealed = true
		statsRevealed = false

		commandRegex = `UPDATE team_users SET view_revenue = \$1, view_stats = \$2, (.+) WHERE team_id = \$3 AND user_id = \$4`
	})

	It("errors when the update fails", func() {
		mock.ExpectExec(commandRegex).
			WithArgs(revenueRevealed, statsRevealed, teamID, channelID).
			WillReturnError(errors.New("some error"))

		err := db.UpdateMembership(context.Background(), teamID, channelID, revenueRevealed, statsRevealed)
		Expect(err).To(HaveOccurred())

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	It("errors when the update affects zero rows", func() {
		mock.ExpectExec(commandRegex).
			WithArgs(revenueRevealed, statsRevealed, teamID, channelID).
			WillReturnResult(sqlmock.NewResult(0, 0))

		err := db.UpdateMembership(context.Background(), teamID, channelID, revenueRevealed, statsRevealed)
		Expect(err).To(HaveOccurred())
		Expect(err).To(Equal(ErrNoMembershipForUpdate))

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})

	It("succeeds when the update affects at least one row", func() {
		rows := int64(1)

		mock.ExpectExec(commandRegex).
			WithArgs(revenueRevealed, statsRevealed, teamID, channelID).
			WillReturnResult(sqlmock.NewResult(0, rows))

		err := db.UpdateMembership(context.Background(), teamID, channelID, revenueRevealed, statsRevealed)
		Expect(err).NotTo(HaveOccurred())

		err = mock.ExpectationsWereMet()
		Expect(err).NotTo(HaveOccurred())
	})
})
