package db

import (
	"context"

	"github.com/pkg/errors"
)

// UpdateTeam makes an UPDATE command to DB to change the editable properties
// of a `teams` record for a given team ID.
func (c *Client) UpdateTeam(ctx context.Context, team Team) error {
	statement := `
		UPDATE teams
 		SET
			display_name = $2,
			user_id = $3,
			info = $4,
			team_logo_image = $5,
			banner_image = $6,
			background_image = $7,
			updated_at = CURRENT_TIMESTAMP
		WHERE id = $1
	`

	var logo *string
	if team.Logo != nil {
		logo = team.Logo.dbString()
	}

	var banner *string
	if team.Banner != nil {
		banner = team.Banner.dbString()
	}

	var background *string
	if team.Background != nil {
		background = team.Background.dbString()
	}

	result, err := c.db.ExecContext(
		ctx,
		statement,
		team.ID,
		team.DisplayName,
		team.UserID,
		team.descriptionHTML,
		logo,
		banner,
		background,
	)

	if err != nil {
		return errors.Wrap(err, "db: failed to update team")
	}

	rowsAffected, err := result.RowsAffected()
	if err != nil {
		return errors.Wrap(err, "db: cannot determine if row update from teams")
	}

	if rowsAffected < 1 {
		return ErrNoTeamForUpdate
	}

	return nil
}
