package db

import (
	"context"
	"database/sql"
	"errors"

	"code.justin.tv/cb/roster/internal/postgres"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"gopkg.in/DATA-DOG/go-sqlmock.v1"
)

var _ = Describe("UpdateTeam", func() {
	var (
		client       *Client
		mock         sqlmock.Sqlmock
		team         Team
		logo         Image
		banner       Image
		background   Image
		commandRegex string
	)

	BeforeEach(func() {
		var stub *sql.DB
		var err error

		stub, mock, err = sqlmock.New()
		Expect(err).NotTo(HaveOccurred())

		client = &Client{
			db: &postgres.DB{
				DB: stub,
			},
		}

		team = Team{
			ID:                  "123",
			DisplayName:         "Staff",
			UserID:              "1234567890",
			descriptionHTML:     "<p>this is team staff</p>",
			descriptionMarkdown: "this is team staff",
		}

		logo = Image{
			ID:     "logo-id",
			Format: "logo-format",
		}

		banner = Image{
			ID:     "banner-id",
			Format: "banner-format",
		}

		background = Image{
			ID:     "background-id",
			Format: "background-format",
		}

		commandRegex = `
			UPDATE teams
			SET
				display_name = \$2,
				user_id = \$3,
				info = \$4,
				team_logo_image = \$5,
				banner_image = \$6,
				background_image = \$7,
				updated_at = CURRENT_TIMESTAMP
			WHERE id = \$1
		`
	})

	Context("when the update fails", func() {
		BeforeEach(func() {
			mock.ExpectExec(commandRegex).
				WithArgs(sqlmock.AnyArg(), sqlmock.AnyArg(), sqlmock.AnyArg(), sqlmock.AnyArg(), sqlmock.AnyArg(), sqlmock.AnyArg(), sqlmock.AnyArg()).
				WillReturnError(errors.New("some error"))
		})

		It("errors when the update fails", func() {
			err := client.UpdateTeam(context.Background(), team)
			Expect(err).To(HaveOccurred())

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})
	})

	Context("when the update affects zero rows", func() {
		BeforeEach(func() {
			mock.ExpectExec(commandRegex).
				WithArgs(sqlmock.AnyArg(), sqlmock.AnyArg(), sqlmock.AnyArg(), sqlmock.AnyArg(), sqlmock.AnyArg(), sqlmock.AnyArg(), sqlmock.AnyArg()).
				WillReturnResult(sqlmock.NewResult(0, 0))
		})

		It("errors when the update affects zero rows", func() {
			err := client.UpdateTeam(context.Background(), team)
			Expect(err).To(HaveOccurred())
			Expect(err).To(Equal(ErrNoTeamForUpdate))

			err = mock.ExpectationsWereMet()
			Expect(err).NotTo(HaveOccurred())
		})
	})

	Context("when the update affects at least one row", func() {
		Context("when the team for update has no images", func() {
			BeforeEach(func() {
				mock.ExpectExec(commandRegex).WithArgs(
					team.ID,
					team.DisplayName,
					team.UserID,
					team.descriptionHTML,
					nil,
					nil,
					nil,
				).WillReturnResult(sqlmock.NewResult(0, int64(1)))
			})

			It("succeeds when the update affects at least one row", func() {
				err := client.UpdateTeam(context.Background(), team)
				Expect(err).NotTo(HaveOccurred())

				err = mock.ExpectationsWereMet()
				Expect(err).NotTo(HaveOccurred())
			})
		})

		Context("when the team for update has images", func() {
			BeforeEach(func() {
				team.Logo = &logo
				team.Banner = &banner
				team.Background = &background

				mock.ExpectExec(commandRegex).WithArgs(
					team.ID,
					team.DisplayName,
					team.UserID,
					team.descriptionHTML,
					":uid: logo-id\n:format: logo-format",
					":uid: banner-id\n:format: banner-format",
					":uid: background-id\n:format: background-format",
				).WillReturnResult(sqlmock.NewResult(0, int64(1)))
			})

			It("succeeds when the update affects at least one row", func() {
				err := client.UpdateTeam(context.Background(), team)
				Expect(err).NotTo(HaveOccurred())

				err = mock.ExpectationsWereMet()
				Expect(err).NotTo(HaveOccurred())
			})
		})
	})
})
