package description

import (
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
)

var _ = Describe("Processors", func() {
	Describe("HTMLToMarkdown", func() {
		It("converts HTML to Markdown while sanitizing malicious inputs", func() {
			html := `
<h1>Hello</h1>
<p><img src="img.jpg" alt="Alt"/></p>
<IMG SRC="jav&#x0D;ascript:alert('XSS');">
<div><img src="data:text/javascript;charset=utf-8,alert('hi');"></div>
<hr/>
<ul>
  <li class="this-class-is-stripped">This is an unordered list A</li>
</ul>
<hr/>
<ol style="background: #000000">
  <li>This is an ordered list B</li>
</ol>
<hr/>
<table border="0" cellspacing="0" cellpadding="0">
  <tbody>
    <tr>
      <td width="400" height="55" colspan="5">
        <div style="text-align:center;">Level|Up Series</div>
      </td>
    </tr>
  </tbody>
</table>
<!-- This comment is ignored -->
<h2 disabled>Links</h2>
<p><a href="http://trusted.org/search.cgi?val=<SCRIPT SRC='http://evil.org/badkama.js'></SCRIPT>">Click here!</a></p>
<h3 id="omitted">Contact</h3>
<P><A href="mailto://bar/">Mail</A></P>
<footer id="useless-white-space">
  <br/>
  <p></p>
  <BR/>
</footer>
`
			expectedMarkdown := "# Hello\n\n" +
				"![Alt](img.jpg)\n\n\n\n\n\n" +
				"* * *\n\n" +
				"*   This is an unordered list A\n\n" +
				"* * *\n\n" +
				"1.  This is an ordered list B\n\n" +
				"* * *\n\n" +
				"\n\n" +
				"  \n\n" +
				"    \n\n" +
				"      \n\n" +
				"        \n" +
				"Level|Up Series\n\n" +
				"      \n\n" +
				"    \n\n" +
				"  \n\n" +
				"\n\n" +
				"\n" +
				"## Links\n\n" +
				"Click here!\n\n" +
				"### Contact\n\n" +
				"[Mail](mailto://bar/)"

			Expect(HTMLToMarkdown(html)).To(Equal(expectedMarkdown))
		})
	})

	Describe("MarkdownToHTML", func() {
		It("converts Markdown to HTML while adding security measures", func() {
			markdown := `# Hello
![Alt](img.jpg)

* This is an unordered list

---

1. This is an ordered list

## Contact
[This is safe!](mailto://bar/)

[This is not safe!](data://malicious/)
`

			expectedHTML := `<h1>Hello</h1>

<p><img src="img.jpg" alt="Alt"/></p>

<ul>
<li>This is an unordered list</li>
</ul>

<hr/>

<ol>
<li>This is an ordered list</li>
</ol>

<h2>Contact</h2>

<p><a href="mailto://bar/" rel="nofollow noopener" target="_blank">This is safe!</a></p>

<p>This is not safe!</p>
`

			Expect(MarkdownToHTML(markdown)).To(Equal(expectedHTML))
		})
	})
})
