package eventbus

import (
	"os"

	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	eventbusClient "code.justin.tv/eventbus/client"
	"code.justin.tv/eventbus/client/subscriber/sqsclient"
	"code.justin.tv/eventbus/schema/pkg/user"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/endpoints"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/pkg/errors"
)

// HandlerParams are the params to instantiate eventbus handlers
type HandlerParams struct {
	Cache      Cache
	DBWriter   DBWriter
	PDMSClient PDMSClient
	Reporter   telemetry.SampleReporter
}

// Handler are the attributes of an Handler
type Handler struct {
	cache    Cache
	dbWriter DBWriter
	pdms     PDMSClient
	reporter telemetry.SampleReporter
}

// Initialise creates an Handler for the worker to use
func Initialise(params *HandlerParams) error {
	mux := eventbusClient.NewMux()

	eventbusHandler := Handler{
		cache:    params.Cache,
		dbWriter: params.DBWriter,
		pdms:     params.PDMSClient,
		reporter: params.Reporter,
	}

	user.RegisterDestroyHandler(mux, eventbusHandler.handleUserDestroy)

	sess, err := session.NewSession(&aws.Config{
		// Current region is pre-defined in the execution environment
		Region:              aws.String(os.Getenv("AWS_REGION")),
		STSRegionalEndpoint: endpoints.RegionalSTSEndpoint,
	})
	if err != nil {
		return errors.Wrap(err, "failed to start AWS session")
	}

	_, err = sqsclient.New(sqsclient.Config{
		Session:    sess,
		Dispatcher: mux.Dispatcher(),
		QueueURL:   os.Getenv("ROSTER_EVENTBUS_SQS_URL"),
	})
	if err != nil {
		return errors.Wrap(err, "sqsclient initialization error")
	}

	return nil
}
