package eventbus

import (
	"context"

	eventbusClient "code.justin.tv/eventbus/client"
	"code.justin.tv/eventbus/schema/pkg/user"
	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"

	"code.justin.tv/cb/roster/internal/db"
)

func (e *Handler) handleUserDestroy(ctx context.Context, h *eventbusClient.Header, event *user.Destroy) error {
	userID := event.GetUserId()

	log.Info("processing hard deletion of user ", userID)

	// 1. Get all memberships
	memberships, err := e.dbWriter.GetChannelMemberships(ctx, userID)
	if err != nil {
		log.WithError(err).WithField("user_id", userID).Error("eventbus: failed to get user memberships")
		return errors.Wrapf(err, "error retrieving membership for user %s", userID)
	}

	// 2. Clear cached memberships
	for _, membership := range memberships {
		go func(teamID string) {
			cacheError := e.cache.ClearAllTeamMembershipsForTeam(context.Background(), teamID)
			if cacheError != nil {
				log.WithError(cacheError).WithField("user_id", userID).Error("worker: failed to clear cached team memberships")
			}
		}(membership.TeamID)
	}

	go func() {
		cacheError := e.cache.ClearChannelMemberships(context.Background(), userID)
		if cacheError != nil {
			log.WithError(cacheError).WithField("user_id", userID).Error("worker: failed to clear cached channel memberships")
		}
	}()

	// 3. Delete all user memberships
	err = e.dbWriter.DeleteMemberships(ctx, userID)
	if err != nil && err != db.ErrNoRowFoundForDeletion {
		log.WithError(err).WithField("user_id", userID).Error("worker: failed to delete memberships for user")
		return errors.Wrapf(err, "error deleting membership for user %s", userID)
	}

	// 4. Delete featured
	err = e.dbWriter.DeleteAllFeaturedByChannel(ctx, userID)
	if err != nil && err != db.ErrNoRowFoundForDeletion {
		log.WithError(err).WithField("user_id", userID).Error("worker: failed to delete featured for user")
		return errors.Wrapf(err, "error deleting featured for user %s", userID)
	}

	// 5. Delete invitations
	err = e.dbWriter.DeleteChannelInvitations(ctx, userID)
	if err != nil && err != db.ErrNoRowFoundForDeletion {
		log.WithError(err).WithField("user_id", userID).Error("worker: failed to delete invitations for user")
		return errors.Wrapf(err, "error deleting invitations for user %s", userID)
	}

	err = e.pdms.PromiseDeletion(ctx, userID)
	if err != nil {
		log.WithError(err).Error("worker: failed to send deletion promise to pdms")
		return errors.Wrapf(err, "error sending deletion promise to pdms for user %s", userID)
	}

	log.Info("successfully processed hard deletion of user ", userID)

	return nil
}
