package eventbus

import (
	"context"
	"testing"

	eventbusClient "code.justin.tv/eventbus/client"
	"code.justin.tv/eventbus/schema/pkg/user"
	"github.com/pkg/errors"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"

	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/eventbus/mocks"
)

func TestUserDestroy(t *testing.T) {
	type UserDestroyInput struct {
		header *eventbusClient.Header
		event  *user.UserDestroy
	}

	type testCase struct {
		description string
		input       UserDestroyInput
		expectedErr error
		setupMocks  func(*mocks.DBWriter, *mocks.Cache, *mocks.PDMSClient)
	}

	ctx := context.Background()
	userID := "test-user-id-123"
	teamID := "test-channel-123"
	testError := errors.Errorf("some error occurred")

	header := &eventbusClient.Header{
		EventType:   "test-event",
		Environment: "dev",
	}

	event := &user.UserDestroy{
		UserId: userID,
		Login:  "test-user-login-123",
	}

	membershipResponse := []db.Membership{
		{
			ChannelID:       userID,
			TeamID:          teamID,
			StatsRevealed:   false,
			RevenueRevealed: false,
		},
	}

	testCases := []testCase{
		{
			description: "process UserDestroy event successfully",
			input: UserDestroyInput{
				header: header,
				event:  event,
			},
			setupMocks: func(dbWriter *mocks.DBWriter, cache *mocks.Cache, pdms *mocks.PDMSClient) {
				dbWriter.On("GetChannelMemberships", mock.Anything, userID).Return(membershipResponse, nil)
				dbWriter.On("DeleteMemberships", mock.Anything, userID).Return(nil)
				cache.On("ClearAllTeamMembershipsForTeam", mock.Anything, teamID).Return(nil)
				cache.On("ClearChannelMemberships", mock.Anything, userID).Return(nil)
				dbWriter.On("DeleteAllFeaturedByChannel", mock.Anything, userID).Return(nil)
				dbWriter.On("DeleteChannelInvitations", mock.Anything, userID).Return(nil)
				pdms.On("PromiseDeletion", mock.Anything, userID).Return(nil)
			},
		},
		{
			description: "fails getting ChannelMemberships for user",
			input: UserDestroyInput{
				header: header,
				event:  event,
			},
			setupMocks: func(dbWriter *mocks.DBWriter, cache *mocks.Cache, pdms *mocks.PDMSClient) {
				dbWriter.On("GetChannelMemberships", mock.Anything, userID).Return(nil, testError)
			},
			expectedErr: testError,
		},
		{
			description: "fails deleting ChannelMemberships for user",
			input: UserDestroyInput{
				header: header,
				event:  event,
			},
			setupMocks: func(dbWriter *mocks.DBWriter, cache *mocks.Cache, pdms *mocks.PDMSClient) {
				dbWriter.On("GetChannelMemberships", mock.Anything, userID).Return(membershipResponse, nil)
				cache.On("ClearAllTeamMembershipsForTeam", mock.Anything, teamID).Return(errors.Errorf("cache error"))
				cache.On("ClearChannelMemberships", mock.Anything, userID).Return(errors.Errorf("cache error"))
				dbWriter.On("DeleteMemberships", mock.Anything, userID).Return(testError)
			},
			expectedErr: testError,
		},
		{
			description: "fails deleting featured channels for user",
			input: UserDestroyInput{
				header: header,
				event:  event,
			},
			setupMocks: func(dbWriter *mocks.DBWriter, cache *mocks.Cache, pdms *mocks.PDMSClient) {
				dbWriter.On("GetChannelMemberships", mock.Anything, userID).Return(membershipResponse, nil)
				dbWriter.On("DeleteMemberships", mock.Anything, userID).Return(nil)
				cache.On("ClearAllTeamMembershipsForTeam", mock.Anything, teamID).Return(nil)
				cache.On("ClearChannelMemberships", mock.Anything, userID).Return(nil)
				dbWriter.On("DeleteAllFeaturedByChannel", mock.Anything, userID).Return(testError)
			},
			expectedErr: testError,
		},
		{
			description: "fails to update PDMS despite successful deletion for user",
			input: UserDestroyInput{
				header: header,
				event:  event,
			},
			setupMocks: func(dbWriter *mocks.DBWriter, cache *mocks.Cache, pdms *mocks.PDMSClient) {
				dbWriter.On("GetChannelMemberships", mock.Anything, userID).Return(membershipResponse, nil)
				dbWriter.On("DeleteMemberships", mock.Anything, userID).Return(nil)
				cache.On("ClearAllTeamMembershipsForTeam", mock.Anything, teamID).Return(nil)
				cache.On("ClearChannelMemberships", mock.Anything, userID).Return(nil)
				dbWriter.On("DeleteAllFeaturedByChannel", mock.Anything, userID).Return(nil)
				dbWriter.On("DeleteChannelInvitations", mock.Anything, userID).Return(nil)
				pdms.On("PromiseDeletion", mock.Anything, userID).Return(testError)
			},
			expectedErr: testError,
		},
		{
			description: "fails to clear cache",
			input: UserDestroyInput{
				header: header,
				event:  event,
			},
			setupMocks: func(dbWriter *mocks.DBWriter, cache *mocks.Cache, pdms *mocks.PDMSClient) {
				dbWriter.On("GetChannelMemberships", mock.Anything, userID).Return(membershipResponse, nil)
				dbWriter.On("DeleteMemberships", mock.Anything, userID).Return(nil)
				cache.On("ClearAllTeamMembershipsForTeam", mock.Anything, teamID).Return(errors.Errorf("cache error"))
				cache.On("ClearChannelMemberships", mock.Anything, userID).Return(errors.Errorf("cache error"))
				dbWriter.On("DeleteAllFeaturedByChannel", mock.Anything, userID).Return(nil)
				dbWriter.On("DeleteChannelInvitations", mock.Anything, userID).Return(nil)
				pdms.On("PromiseDeletion", mock.Anything, userID).Return(nil)
			},
		},
		{
			description: "fails deleting invitation for user",
			input: UserDestroyInput{
				header: header,
				event:  event,
			},
			setupMocks: func(dbWriter *mocks.DBWriter, cache *mocks.Cache, pdms *mocks.PDMSClient) {
				dbWriter.On("GetChannelMemberships", mock.Anything, userID).Return(membershipResponse, nil)
				dbWriter.On("DeleteMemberships", mock.Anything, userID).Return(nil)
				cache.On("ClearAllTeamMembershipsForTeam", mock.Anything, teamID).Return(nil)
				cache.On("ClearChannelMemberships", mock.Anything, userID).Return(nil)
				dbWriter.On("DeleteAllFeaturedByChannel", mock.Anything, userID).Return(nil)
				dbWriter.On("DeleteChannelInvitations", mock.Anything, userID).Return(testError)
			},
			expectedErr: testError,
		},
		{
			description: "user was already deleted from our systems, but we are still able to report a deletion",
			input: UserDestroyInput{
				header: header,
				event:  event,
			},
			setupMocks: func(dbWriter *mocks.DBWriter, cache *mocks.Cache, pdms *mocks.PDMSClient) {
				dbWriter.On("GetChannelMemberships", mock.Anything, userID).Return(membershipResponse, nil)
				dbWriter.On("DeleteMemberships", mock.Anything, userID).Return(db.ErrNoRowFoundForDeletion)
				cache.On("ClearAllTeamMembershipsForTeam", mock.Anything, teamID).Return(errors.Errorf("entry not found in cache"))
				cache.On("ClearChannelMemberships", mock.Anything, userID).Return(errors.Errorf("entry not found in cache"))
				dbWriter.On("DeleteAllFeaturedByChannel", mock.Anything, userID).Return(db.ErrNoRowFoundForDeletion)
				dbWriter.On("DeleteChannelInvitations", mock.Anything, userID).Return(db.ErrNoRowFoundForDeletion)
				pdms.On("PromiseDeletion", mock.Anything, userID).Return(nil)
			},
		},
	}

	for _, testCase := range testCases {
		dbWriter := &mocks.DBWriter{}
		cache := &mocks.Cache{}
		pdms := &mocks.PDMSClient{}

		eventBusHandler := Handler{
			dbWriter: dbWriter,
			cache:    cache,
			pdms:     pdms,
		}

		if testCase.setupMocks != nil {
			testCase.setupMocks(dbWriter, cache, pdms)
		}

		t.Run(testCase.description, func(t *testing.T) {
			actualErr := eventBusHandler.handleUserDestroy(ctx, testCase.input.header, testCase.input.event)
			if testCase.expectedErr != nil {
				assert.Equal(t, errors.Cause(testError), errors.Cause(actualErr))
			} else {
				require.NoError(t, testCase.expectedErr)
			}
		})

		dbWriter.AssertExpectations(t)
		cache.AssertExpectations(t)
		pdms.AssertExpectations(t)
	}
}
