package httputil

import (
	"encoding/json"
	"fmt"
	"math"
	"net/http"
	"time"

	log "github.com/sirupsen/logrus"
)

// JSONResponseWriter is used to write JSON response for all HTTP requests.
type JSONResponseWriter struct {
	http.ResponseWriter
}

// NewJSONResponseWriter returns a JSONResponseWriter for the given request.
func NewJSONResponseWriter(w http.ResponseWriter) *JSONResponseWriter {
	return &JSONResponseWriter{w}
}

// Cacheable sets the Cache-Control header in the response to indicate
// to the client that the response payload is cacheable for the given duration.
func (w *JSONResponseWriter) Cacheable(max time.Duration) {
	wholeSeconds := int(math.Floor(max.Seconds()))

	if wholeSeconds == 0 {
		return
	}

	w.Header().Add("Cache-Control", "public")
	w.Header().Add("Cache-Control", fmt.Sprintf("max-age=%d", wholeSeconds))
}

// OK returns a 200 status code and serializes the given structure as JSON.
func (w *JSONResponseWriter) OK(data interface{}) {
	w.serveJSON(http.StatusOK, data)
}

// OKRaw returns a 200 status code and writes raw bytes into response.
func (w *JSONResponseWriter) OKRaw(bytes []byte) {
	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(http.StatusOK)

	_, err := w.Write(bytes)
	if err != nil {
		log.WithError(err).Error("failed to write http response")
	}
}

// Created returns a 201 status code and serializes the given structure as JSON.
func (w *JSONResponseWriter) Created(data interface{}) {
	w.serveJSON(http.StatusCreated, data)
}

// BadRequest returns a 400 status code.
func (w *JSONResponseWriter) BadRequest(message interface{}) {
	w.serveError(http.StatusBadRequest, message)
}

// Unauthorized returns a 401 status code.
func (w *JSONResponseWriter) Unauthorized(message interface{}) {
	w.serveError(http.StatusUnauthorized, message)
}

// Forbidden returns a 403 status code.
func (w *JSONResponseWriter) Forbidden(message interface{}) {
	w.serveError(http.StatusForbidden, message)
}

// NotFound returns a 404 status code.
func (w *JSONResponseWriter) NotFound(message interface{}) {
	w.serveError(http.StatusNotFound, message)
}

// Conflict returns a 409 status code.
func (w *JSONResponseWriter) Conflict(message interface{}) {
	w.serveError(http.StatusConflict, message)
}

// Gone returns a 410 status code.
func (w *JSONResponseWriter) Gone(message interface{}) {
	w.serveError(http.StatusGone, message)
}

// UnprocessableEntity returns a 422 status code.
func (w *JSONResponseWriter) UnprocessableEntity(message interface{}) {
	w.serveError(http.StatusUnprocessableEntity, message)
}

// InternalServerError returns a 500 status code and logs the error.
func (w *JSONResponseWriter) InternalServerError(message interface{}, err error) {
	status := http.StatusInternalServerError

	log.WithError(err).Error(http.StatusText(status))
	w.serveError(status, message)
}

// ServiceUnavailable returns a 503 status code and logs the error.
func (w *JSONResponseWriter) ServiceUnavailable(message interface{}, err error) {
	status := http.StatusServiceUnavailable

	log.WithError(err).Error(http.StatusText(status))
	w.serveError(status, message)
}

func (w *JSONResponseWriter) serveJSON(status int, data interface{}) {
	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(status)

	err := json.NewEncoder(w).Encode(data)
	if err != nil {
		status = http.StatusInternalServerError

		log.WithError(err).Error("Failed to write HTTP JSON response")
		http.Error(w, http.StatusText(status), status)
	}
}

func (w *JSONResponseWriter) serveError(status int, message interface{}) {
	if message != nil {
		message = fmt.Sprintf("%s", message)
	}

	response := map[string]interface{}{
		"error":   http.StatusText(status),
		"status":  status,
		"message": message,
	}

	w.Header().Del("Cache-Control")
	w.serveJSON(status, response)
}
