package integration

import (
	"context"
	"net/http/httptest"

	v1 "code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/client/roster"
	"code.justin.tv/cb/roster/internal/api"
	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/authorization"
	"code.justin.tv/cb/roster/internal/cache"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var (
	mockedCache *mocks.Cache
	dbReader    *mocks.DBReader
	server      *httptest.Server
	client      roster.Client
	err         error
)

var _ = Describe("GetV1TeamsClient", func() {
	BeforeEach(func() {
		mockedCache = &mocks.Cache{}
		dbReader = &mocks.DBReader{}

		rosterServer := api.NewServer(&api.ServerParams{
			AuthDecoder:      &authorization.Decoder{},
			Cache:            mockedCache,
			DBWriter:         &mocks.DBWriter{},
			DBReader:         dbReader,
			Users:            &mocks.Users{},
			TelemetryHandler: &telemetryhook.NoopClient{},
		})

		server = httptest.NewServer(rosterServer)
		client, err = roster.NewClient(twitchclient.ClientConf{
			Host: server.URL,
		})
		Expect(err).NotTo(HaveOccurred())
	})

	AfterEach(func() {
		server.Close()
	})

	Context("When querying for all teams (not cached)", func() {
		teamID := "123"
		teamName := "staff"
		teamOwnerID := "emmett"
		total := uint(9000)
		limit := uint(db.GetTeamsDefaultLimit)

		BeforeEach(func() {
			cacheQuery := cache.TeamsQuery{
				Limit:            &limit,
				OrderByDirection: db.OrderByDirectionAsc,
			}

			mockedCache.On("GetTeams", mock.Anything, cacheQuery).Return("", cache.ErrNoTeams)

			filter := db.TeamsFilter{
				Limit:            &limit,
				OrderByDirection: db.OrderByDirectionAsc,
			}

			dbReader.On("GetTeamsCount", mock.Anything, filter).Return(total, nil)

			dbReader.On("GetTeams", mock.Anything, filter).Return([]db.Team{
				{
					ID:     teamID,
					Name:   teamName,
					UserID: teamOwnerID,
				},
			}, nil)

			mockedCache.On("SetTeams", mock.Anything, cacheQuery, mock.Anything).Return(nil)
		})

		It("succeeds", func() {
			resp, err := client.GetV1Teams(context.Background(), v1.GetTeamsRequestParams{}, nil)
			Expect(err).NotTo(HaveOccurred())

			Expect(resp).NotTo(BeNil())
			Expect(resp.Meta.Limit).To(Equal(uint(db.GetTeamsDefaultLimit)))
			Expect(resp.Meta.Offset).To(Equal(uint(0)))
			Expect(resp.Meta.Total).To(Equal(total))
			Expect(resp.Data).To(HaveLen(1))
			Expect(resp.Data[0].ID).To(Equal(teamID))
			Expect(resp.Data[0].Name).To(Equal(teamName))
			Expect(resp.Data[0].UserID).To(Equal(teamOwnerID))
		})
	})

	Context("When querying with query parameters (not cached)", func() {
		teamID := "123"
		teamName := "staff"
		teamOwnerID := "emmett"
		limit := uint(99)
		offset := uint(333)
		sort := v1.SortDesc
		total := uint(9000)

		BeforeEach(func() {
			cacheQuery := cache.TeamsQuery{
				Limit:            &limit,
				Name:             teamName,
				Offset:           offset,
				OrderByDirection: db.OrderByDirectionDesc,
				UserID:           teamOwnerID,
			}

			mockedCache.On("GetTeams", mock.Anything, cacheQuery).Return("", cache.ErrNoTeams)

			filter := db.TeamsFilter{
				Limit:            &limit,
				Name:             teamName,
				Offset:           offset,
				OrderByDirection: db.OrderByDirectionDesc,
				UserID:           teamOwnerID,
			}

			dbReader.On("GetTeamsCount", mock.Anything, filter).Return(total, nil)

			dbReader.On("GetTeams", mock.Anything, filter).Return([]db.Team{
				{
					ID:     teamID,
					Name:   teamName,
					UserID: teamOwnerID,
				},
			}, nil)

			mockedCache.On("SetTeams", mock.Anything, cacheQuery, mock.Anything).Return(nil)
		})

		It("succeeds", func() {
			params := v1.GetTeamsRequestParams{
				Limit:  limit,
				Name:   teamName,
				Offset: offset,
				Sort:   sort,
				UserID: teamOwnerID,
			}

			resp, err := client.GetV1Teams(context.Background(), params, nil)
			Expect(err).NotTo(HaveOccurred())

			Expect(resp).NotTo(BeNil())
			Expect(resp.Meta.Limit).To(Equal(limit))
			Expect(resp.Meta.Offset).To(Equal(offset))
			Expect(resp.Meta.Sort).To(Equal(sort))
			Expect(resp.Meta.Total).To(Equal(total))
			Expect(resp.Data).To(HaveLen(1))
			Expect(resp.Data[0].ID).To(Equal(teamID))
			Expect(resp.Data[0].Name).To(Equal(teamName))
			Expect(resp.Data[0].UserID).To(Equal(teamOwnerID))
		})
	})
})
